# -*- coding: utf-8 -*-

"""
Collection d'objets flask.jsonify représentant des erreurs HTTP
qu'on peut implémenter dans ses APIs très facilement.

Cette collection permet de standardiser le format des erreurs en JSON
suivant cette syntaxe:
{
    "status": le status code http en entier,
    "code": le code d'erreur propre à l'API, en chaine de caractères,
    "error": un message court décrivant l'erreur,
    "message": un message détaillant l'erreur
}
"""

from flask import jsonify

from .. import LDAP_API

import os


DEFAULTS = {
    '500': (
        'erreur interne',
        'Une erreur interne est survenue'
        ),
    '502': (
        'serveur injoignable',
        'Le serveur est injoignable'
        ),
    '503': (
        'service indisponible',
        'Service temporairement indisponible ou en maintenance'
        ),
    '504': (
        'timeout',
        'Le serveur met trop de temps à répondre'
        ),
    '400': (
        'requête incorrecte',
        'La syntaxe de la requête est erronée'
        ),
    '401': (
        'authentification requise',
        'Une authentification est nécessaire pour accéder à la ressource'
        ),
    '403': (
        'interdit',
        "Vous n'avez pas les droits nécessaires"
        ),
    '404': (
        'introuvable',
        'Ressource introuvable'
        ),
    '405': (
        'non autorisé',
        'Méthode de requête non autorisée'
        ),
    '409': (
        'conflit',
        'La requête ne peut être traitée en l’état actuel'
        ),
    '410': (
        'indisponible',
        'La ressource est indisponible'
        )
    }


class HTTPError(object):
    """Constructeur d'erreurs HTTP standards en JSON"""

    def __init__(self, status, service_code=None):
        self.status = str(status)  # status HTTP
        # code représentant le service implémentant l'erreur
        # s'il n'est pas précisé, la classe tente de récupérer
        # une valeur dans les variables d'environnement;
        # si même là ça échoue alors un code standard est utilisé
        self.service_code = (
            service_code or
            os.environ.get('SERVICE_CODE', '04x-yy')
            )

    def response(self, message=None, code=None, request_id=None):
        """Fonction de création de l'objet JSON"""

        if not code:
            # le code par défaut est une combinaison entre le service_code
            # et le status HTTP
            code = '{root}-{suffix}'.format(
                root=self.service_code,
                suffix=self.status
                )
        response = jsonify(
            {
                'status': int(self.status),
                'code': code,
                'error': DEFAULTS[self.status][0],
                'message': message or DEFAULTS[self.status][1]
                }
            )
        response.status_code = int(self.status)
        if self.status == '401':
            response.headers = {
                'WWW-Authenticate': 'Basic realm="Authentication required"',
                'Content-Type': 'application/json',
                'Location': '{}/auth/request-token'.format(LDAP_API)
                }
        else:
            response.headers = {
                'Content-Type': 'application/json'
                }
        if request_id:
            response.headers['X-Request-Id'] = request_id

        return response


def internal_error(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 500"""
    return HTTPError(500, service_code).response(message, code, request_id)


def bad_gateway(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 502"""
    return HTTPError(502, service_code).response(message, code, request_id)


def service_unavailable(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 503"""
    return HTTPError(503, service_code).response(message, code, request_id)


def gateway_timeout(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 504"""
    return HTTPError(504, service_code).response(message, code, request_id)


def bad_request(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 400"""
    return HTTPError(400, service_code).response(message, code, request_id)


def unauthorized(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 401"""
    return HTTPError(401, service_code).response(message, code, request_id)


def forbidden(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 403"""
    return HTTPError(403, service_code).response(message, code, request_id)


def not_found(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 404"""
    return HTTPError(404, service_code).response(message, code, request_id)


def not_allowed(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 405"""
    return HTTPError(405, service_code).response(message, code, request_id)


def conflict(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 409"""
    return HTTPError(409, service_code).response(message, code, request_id)


def gone(
        message=None,
        code=None,
        request_id=None,
        service_code=None
        ):
    """HTTP 410"""
    return HTTPError(410, service_code).response(message, code, request_id)

# EOF
