#!/usr/bin/env python
# encoding: utf-8

from flask import jsonify, request, Response

from blueflask.lib.infos import no_content

from ...channel import Channel
from ...program import ProgramGuide
from . import api

import datetime
import xmltodict


@api.route('/programs/<int:service_id>', methods=['GET'])
def get_weekly_program_guide(service_id):
    """
    Guide des programmes d'une chaine pour la semaine en cours.
    Cette ressource renvoie le guide complet des programmes d'une chaine
    pour la semaine en cours.
    ---
    tags:
      - EPG
    definitions:
      - schema:
          id: Chaine TV
          properties:
            name:
              type: string
              description: Nom de la chaine
            service_id:
              type: string
              description: Numéro de la chaine sur le réseau blueline TV
      - schema:
          id: Programme
          properties:
            title:
              type: string
              description: Titre de l'émission
            description:
              type: string
              description: Synopsis de l'émission
            start:
              type: string
              description: Début de l'émission
            stop:
              type: string
              description: Fin de l'émission
      - schema:
          id: Dates de diffusion
          properties:
            length:
              type: integer
              description: Nombre de jours avec programmes existant
            service:
              type: "#/definitions/Chaine TV"
              description: Chaine TV
            dates:
              type: array
              description: Liste des dates de diffusion
              items:
                properties:
                    date:
                    programs:
                      type: array
                      items: "#/definitions/Programme"
    responses:
      200:
        description: Dates de diffusion
        type: "#/definitions/Dates de diffusion"
    """
    channel = Channel(service_id)
    today = datetime.date.today()
    current_weekday = today.weekday()
    dates = [
        today + datetime.timedelta(i-current_weekday)
        for i in range(7)
        ]
    program_guides = [
        ProgramGuide(service_id, date.strftime('%d%m%Y'))
        for date in dates
        ]
    for program_guide in program_guides:
        program_guide.get_tmira_data()
    program_guides = [
        program_guide for program_guide in program_guides
        if program_guide.guide
        ]
    result = {
        'length': len(dates),
        'service': {
            'name': channel.name,
            'service_id': service_id
            },
        'dates': [
            {
                'date': program_guide.date.strftime('%d%m%Y'),
                'programs': [
                    {
                        'title': program.title,
                        'description': program.description,
                        'start': program.start.strftime('%d%m%Y %H:%M:%S'),
                        'stop': program.stop.strftime('%d%m%Y %H:%M:%S')
                        }
                    for program in program_guide.guide
                    ]
                }
            for program_guide in program_guides
            ]
        }
    return jsonify(result)


@api.route('/programs/<int:service_id>/<string:date>', methods=['GET'])
def get_program_guide(service_id, date):
    """
    Guide des programmes d'une chaine pour une date donnée.
    Cette ressource renvoie le guide complet des programmes d'une chaine
    pour une date précise.
    ---
    tags:
      - EPG
    definitions:
      - schema:
          id: Chaine TV
          properties:
            name:
              type: string
              description: Nom de la chaine
            service_id:
              type: string
              description: Numéro de la chaine sur le réseau blueline TV
      - schema:
          id: Programme
          properties:
            title:
              type: string
              description: Titre de l'émission
            description:
              type: string
              description: Synopsis de l'émission
            start:
              type: string
              description: Début de l'émission
            stop:
              type: string
              description: Fin de l'émission
      - schema:
          id: Guide des programmes
          properties:
            length:
              type: integer
              description: Nombre d'émission prévues dans la journée
            service:
              type: "#/definitions/Chaine TV"
              description: Chaine TV
            date:
              type: string
              description: Date du début de diffusion du guide
            programs:
              type: array
              items: "#/definitions/Programme"
    responses:
      200:
        description: Guide des programmes
        type: "#/definitions/Guide des programmes"
    """
    program_guide = ProgramGuide(service_id, date)
    if 'xml' in request.args:
        xml = True
    else:
        xml = False
    program_guide.get_tmira_data()
    if not program_guide.guide:
        return no_content()
    elif not xml:
        programs = program_guide.guide
        result = {
            'length': len(programs),
            'date': date,
            'service': {
                'name': program_guide.channel.name,
                'service_id': service_id
                },
            'programs': [
                {
                    'title': program.title,
                    'description': program.description,
                    'start': program.start.strftime('%d%m%Y %H:%M:%S'),
                    'stop': program.stop.strftime('%d%m%Y %H:%M:%S')
                    }
                for program in programs
                ]
            }
        return jsonify(result)
    else:
        programs = program_guide.guide_xmltv
        result = xmltodict.unparse(programs)
        return Response(result, mimetype='text/xml')


@api.route('/programs/<int:service_id>/current', methods=['GET'])
def get_current_program(service_id):
    """
    Retourne le programme en cours de diffusion sur la chaine.
    ---
    tags:
      - EPG
    definitions:
      - schema:
          id: Chaine TV
          properties:
            name:
              type: string
              description: Nom de la chaine
            service_id:
              type: string
              description: Numéro de la chaine sur le réseau blueline TV
      - schema:
          id: Programme
          properties:
            title:
              type: string
              description: Titre de l'émission
            description:
              type: string
              description: Synopsis de l'émission
            start:
              type: string
              description: Début de l'émission
            stop:
              type: string
              description: Fin de l'émission
    responses:
      200:
        description: Programme en cours de diffusion
        type:
          properties:
            service:
              type: "#/definitions/Chaine TV"
            program:
              type: "#/definitions/Programme"
    """
    now = datetime.datetime.now()
    seconds = (3600 * 3) + (60 * 0)
    offset = datetime.timedelta(0, seconds)
    tz = datetime.timezone(offset)
    now = now.replace(tzinfo=tz)
    program_guide = ProgramGuide(service_id, now.strftime('%d%m%Y'))
    program_guide.get_tmira_data()
    programs = [
        program for program in program_guide.guide
        if program.start <= now
        ]
    if not programs:
        return no_content()
    else:
        program = programs[-1]
        result = {
            'service': {
                'name': program.channel.name,
                'service_id': service_id
                },
            'program': {
                'title': program.title,
                'description': program.description,
                'start': program.start.strftime('%d%m%Y %H:%M:%S'),
                'stop': program.stop.strftime('%d%m%Y %H:%M:%S')
                }
            }
        return jsonify(result)
# EOF
