#!/usr/bin/env python
# encoding: utf-8


from itertools import groupby
from zipfile import ZipFile

from .channel import Channel
from .consts import CONFIG, LOGS
from .program import Program, ProgramGuide

import datetime
import os
import time
import xmltodict
import wget
import collections

__all__ = [
    'EpgSource'
    ]


class EpgSource(object):
    """
    Représente une source EPG qui procure un document .zip
    contenant les guides des programmes de plusieurs chaines
    sur plusieurs jours
    """

    def __init__(self, name):
        self.name = name
        self.url = CONFIG['URLS'][self.name]
        self.xmlfile = os.path.join(
            CONFIG['FOLDERS']['DOWNLOAD_FOLDER'],
            'tvguide_{}.xml'.format(self.name)
            )

    @property
    def channels(self):
        channel_sections = [
            section for section in CONFIG.sections()
            if section.startswith('CHANNEL_')
            ]
        result = []
        for section in channel_sections:
            _, service_id = section.split('_')
            if CONFIG[section].get(self.name):
                result.append(Channel(service_id))
        return result

    def get_channel(self, name):
        for channel in self.channels:
            section = 'CHANNEL_{}'.format(channel.service_id)
            value = CONFIG[section].get(self.name)
            if value == name:
                return channel
        else:
            return None

    def download(self):

        output = os.path.join(
            CONFIG['FOLDERS']['DOWNLOAD_FOLDER'],
            'tvguide_{}.zip'.format(self.name)
            )
        wget.download(
            self.url,
            out=output
            )

        tvguide = ZipFile(output)

        xmlfile = [
            i for i in tvguide.infolist()
            if i.filename.endswith('.xml')
            ].pop()
        tvguide.extract(xmlfile, path=CONFIG['FOLDERS']['DOWNLOAD_FOLDER'])
        os.rename(
            os.path.join(
                CONFIG['FOLDERS']['DOWNLOAD_FOLDER'],
                xmlfile.filename
                ),
            self.xmlfile
            )
        os.remove(output)
        return True

    def parse(self):

        with open(self.xmlfile) as f:
            xmltv = xmltodict.parse(f.read())
        programs = xmltv['tv'].get('programme', [])
        programs = [i for i in programs if self.get_channel(i['@channel'])]
        programs = sorted(programs, key=lambda x: x['@channel'])
        # Verifier si le title a une lang
        pg = []
        for i in programs:
            if type(dict(i)['title']) == list:
                i = dict(i)
                i['title'] = i['title'][0]
                i = collections.OrderedDict(i)
            if type(dict(i)['title']) != collections.OrderedDict:
                i = dict(i)
                i['title'] = collections.OrderedDict(
                    [('@lang', 'fr'), ('#text', i['title'])]
                            )
                i = collections.OrderedDict(i)
            pg.append(i)
        programs = pg
        # les programmes sont maintenant ordonnés suivant l'identifiant
        # attribué par le service
        programs = {
            identifier: list(infos)
            for identifier, infos in groupby(
                    programs,
                    key=lambda x: x['@channel']
                    )
            }
        # on corrige les infos des programmes retardés/avancés
        for channel in self.channels:
            identifier = getattr(channel, self.name)
            if hasattr(channel, 'delay') and identifier in programs:
                infos = programs[identifier]
                seconds = 3600 * float(channel.delay)
                for info in infos:
                    tz = info['@start'].split()[1]
                    start = time.mktime(
                        time.strptime(
                            info['@start'], '%Y%m%d%H%M%S %z'
                            )
                        )
                    stop = time.mktime(
                        time.strptime(
                            info['@stop'], '%Y%m%d%H%M%S %z'
                            )
                        )
                    info['@start'] = datetime.datetime.fromtimestamp(
                        start+seconds).strftime('%Y%m%d%H%M%S')
                    info['@stop'] = datetime.datetime.fromtimestamp(
                        stop+seconds).strftime('%Y%m%d%H%M%S')
                    info['@start'] = '{start} {tz}'.format(
                        start=info['@start'],
                        tz=tz
                        )
                    info['@stop'] = '{stop} {tz}'.format(
                        stop=info['@stop'],
                        tz=tz
                        )
                programs[identifier] = infos

        for identifier, infos in programs.items():
            programs[identifier] = sorted(
                infos,
                key=lambda x: datetime.datetime.strptime(
                    x['@start'],
                    '%Y%m%d%H%M%S %z'
                    )
                )
        # chaque lot de programme, indexé par identifiant contient la
        # programmation sur X jours
        result = []
        for identifier, programs_list in programs.items():
            channel = self.get_channel(identifier)
            foo = groupby(
                programs_list,
                key=lambda x: datetime.datetime.strptime(
                    x['@start'],
                    '%Y%m%d%H%M%S %z'
                    ).date()
                )
            for day, infos in foo:
                program_guide = ProgramGuide(
                    channel.service_id,
                    day.strftime('%d%m%Y')
                    )
                program_guide.guide = [
                    Program(
                        channel.service_id,
                        datetime.datetime.strptime(
                            info['@start'],
                            '%Y%m%d%H%M%S %z'
                            ).strftime('%d%m%Y %H:%M:%S %z'),
                        datetime.datetime.strptime(
                            info['@stop'],
                            '%Y%m%d%H%M%S %z'
                            ).strftime('%d%m%Y %H:%M:%S %z'),
                        info['title'],
                        info.get('desc', None),
                        credit=info.get('credits', None)
                        )
                    for info in list(infos)
                    ]
                result.append(program_guide)
        return result

    def write_to_ftp(self):
        program_guides = self.parse()
        for program_guide in program_guides:
            channel = program_guide.channel
            date = program_guide.date.strftime('%Y%m%d')
            for program in program_guide.guide:
                program.tzconv()
            xmltv = program_guide.guide_xmltv
            LOGS.logger.info(xmltv)
            folder = os.path.join(
                CONFIG['FOLDERS']['DOWNLOAD_FOLDER'],
                channel.service_id
                )
            try:
                os.listdir(folder)
            except OSError:
                os.mkdir(folder)
            filepath = os.path.join(folder, date+'.xml')
            print(filepath)
            with open(filepath, 'w+b') as f:
                try:
                    xmltodict.unparse(xmltv, output=f)
                except Exception:
                    print("Unparsable file "+str(filepath))
                LOGS.logger.info("{} enregistré".format(program_guide))

# EOF
