#!/usr/bin/env python
# encoding: utf-8

from datetime import datetime, timedelta
from itertools import groupby, repeat

from . import CONFIG
from . import DB
from . import LOGS
from . import REST_CLIENT

__all__ = [
    'Customer',
    ]


class Customer(object):
    """Un client internet est identifié par un nom, un refnum et un produit"""

    def __init__(self, name):
        infos = DB.hgetall('customer:{name}'.format(name=name))
        if not infos:
            raise ValueError(
                "Le client {name} n'existe pas".format(name=name)
                )
        else:
            infos = {
                key.decode('utf-8'): value.decode('utf-8')
                for key, value in infos.items()
                }
            self.name = name
            from .product import Product
            self.product = Product(infos['product'])
            self.refnum = infos['refnum']

    def __repr__(self):
        return '<Customer {name}>'.format(name=self.name)

    @staticmethod
    def all():
        customers = [
            Customer(name.decode('utf-8').split(':')[1])
            for name in DB.keys('customer:*')
            ]
        return customers

    @property
    def fup_status(self):
        """Retourne le status FUP du client"""
        result = [
            {
                'interval': key.decode('utf-8').split(':')[-1],
                'status': eval(DB.hget(key, 'status'))
                }
            for key in DB.keys('fup:{name}:*')
            ]
        return result

    def get_cdr_from_aiguillier(self):
        """Récupération des CDRs des 30 derniers jours
        depuis l'API d'Aiguillier"""
        response = REST_CLIENT.get(
            '/fup',
            auth=(
                CONFIG['AIGUILLIER']['username'],
                CONFIG['AIGUILLIER']['password']
                ),
            params={'customer': self.name}
            )
        infos = response['content']
        if 'refnum' not in infos:  # le client n'existe plus
            keys = DB.keys('*{name}*'.format(name=self.name))
            for key in keys:
                DB.delete(key)
            return None
        else:
            cdrs = sorted(
                infos.get('cdr', []),
                key=lambda x: datetime.strptime(
                    x['sessiontime'],
                    '%Y-%m-%dT%H:%M:%S'
                    )
                )
            cdrs = groupby(
                cdrs,
                key=lambda x: datetime.strptime(
                    x['sessiontime'].split('T')[0],
                    '%Y-%m-%d'
                    )
                )
            result = {}
            for date, infos in cdrs:
                result[date.strftime('%d%m%Y')] = []
                for info in infos:
                    time = info['sessiontime'].split('T')[1].replace(':', '')
                    key = 'cdr:{customer}:{date}:{time}:{source}'.format(
                        customer=self.name,
                        date=date.strftime('%d%m%Y'),
                        time=time,
                        source=info['source']
                        )
                    values = {
                        'octets_in': info['octetsin'],
                        'octets_out': info['octetsout']
                        }
                    DB.hmset(key, values)
                    values['time'] = datetime.strptime(time, '%H%M%S')
                    values['source'] = info['source']
                    values['date'] = date
                    result[date.strftime('%d%m%Y')].append(values)
            return result

    def get_cdr(self, start=None, stop=None):
        """Récupération des données de conso d'un client depuis Redis.
        Les dates sont des objets datetime.datetime
        """
        if not start:
            start = (datetime.now() - timedelta(30)).date()
        if not stop:
            stop = (datetime.now() - timedelta(1)).date()
        if stop < start:
            raise ValueError(
                "La date de fin est inférieure à la date de début"
                )
        LOGS.logger.info(
            "Récupération des CDRS de {customer} du {start} au {stop}".format(
                customer=self.name,
                start=start.strftime('%d%m%Y'),
                stop=stop.strftime('%d%m%Y')
                )
            )
        cdrs = {}
        keys = DB.keys(
            'cdr:{customer}:*'.format(
                customer=self.name
                )
            )
        keys = sorted(
            keys,
            key=lambda x: datetime.strptime(
                '{} {}'.format(
                    x.decode('utf-8').split(':')[2],
                    x.decode('utf-8').split(':')[3],
                    ),
                '%d%m%Y %H%M%S'
                )
            )
        keys = groupby(
            keys,
            key=lambda x: datetime.strptime(
                x.decode('utf-8').split(':')[2],
                '%d%m%Y'
                ).date()
            )
        keys = {i: list(j) for i, j in keys}

        date = start
        while date <= stop:
            _keys = keys.get(date, [])
            infos = []
            for key in _keys:
                key = key.decode('utf-8')
                time, source = key.split(':', 3)[-1].split(':')
                octets_in = DB.hget(key, 'octets_in')
                octets_out = DB.hget(key, 'octets_out')
                infos.append(
                    {
                        'time': datetime.strptime(time, '%H%M%S'),
                        'octets_in': float(octets_in),
                        'octets_out': float(octets_out),
                        'source': source,
                        'date': date
                        }
                    )
            cdrs[date.strftime('%d%m%Y')] = infos
            date += timedelta(1)
        return cdrs

    def filter_cdrs_policy_wise(self, cdrs):
        """Suivant la politique de bon usage du produit réorganiser les cdrs"""
        def time_in_interval(interval, info):
            time = info['time']
            day = info['date'].isoweekday()
            start_time = interval['start']
            stop_time = interval['stop']
            in_days = (interval['day_start'] <= day <= interval['day_stop'])
            if stop_time >= start_time:
                in_times = (start_time <= time <= stop_time)
            else:
                in_times = (start_time <= time or time <= stop_time)
            return (in_times and in_days)
        result = []
        for policy_name, policy in self.product.policies.items():
            info = {
                'data_limit': policy['data_limit'],
                'interval': policy['interval'],
                'name': policy_name,
                'total': 0,
                'octets_in': 0,
                'octets_out': 0,
                'details': []
                }
            intervals = []
            for interval in info['interval'].split('_'):
                start, stop, day_start, day_stop = interval.split(',')
                intervals.append(
                    {
                        'start': datetime.strptime(start, '%H%M%S'),
                        'stop': datetime.strptime(stop, '%H%M%S'),
                        'day_start': int(day_start),
                        'day_stop': int(day_stop)
                        }
                    )
            for day, infos in cdrs.items():
                data = [
                    i for i in infos
                    if any(map(time_in_interval, intervals, repeat(i)))
                    ]
                octets_in = sum([i['octets_in'] for i in data])
                octets_out = sum([i['octets_out'] for i in data])
                total = octets_in + octets_out
                info['total'] += total
                info['octets_in'] += octets_in
                info['octets_out'] += octets_out
                info['details'].append(
                    {
                        'date': day,
                        'octets_in': octets_in,
                        'octets_out': octets_out,
                        'total': total
                        }
                    )
            result.append(info)
        return result

    def apply_product_policies(self, for_real=False, from_db=False):
        """Application des politiques de bon usage du produit"""
        if from_db:
            cdrs = self.get_cdr()
        else:
            cdrs = self.get_cdr_from_aiguillier()
        cdrs_by_policies = self.filter_cdrs_policy_wise(cdrs)
        for info in cdrs_by_policies:
            total = info['total']
            octets_in = info['octets_in']
            octets_out = info['octets_out']
            info['total'] = round(total / 1000000000, 2)
            info['octets_in'] = round(octets_in / 1000000000, 2)
            info['octets_out'] = round(octets_out / 1000000000, 2)
            key = 'fup:{name}:{bar}'.format(name=self.name, bar=info['name'])
            status = eval(DB.hget(key, 'status'))
            if info['total'] >= info['data_limit'] and not status:
                info['fup_in'] = True
            elif info['total'] < info['data_limit'] and status:
                info['fup_out'] = True

            if info.get('fup_in'):  # à limiter
                DB.hmset(
                    key,
                    {
                        'status': True,
                        datetime.today().strftime('%d%m%Y'): 'in'
                        }
                    )
                if for_real:  # demander à Aiguillier de le faire
                    REST_CLIENT.post(
                        '/fup',
                        auth=(
                            CONFIG['AIGUILLIER']['username'],
                            CONFIG['AIGUILLIER']['password']
                            ),
                        params={'customer': self.name}
                        )
            elif info.get('fup_out'):  # libérer
                DB.hmset(
                    key,
                    {
                        'status': False,
                        datetime.today().strftime('%d%m%Y'): 'out'
                        }
                    )
                if for_real:  # demander à Aiguillier de le faire
                    REST_CLIENT.delete(
                        '/fup',
                        auth=(
                            CONFIG['AIGUILLIER']['username'],
                            CONFIG['AIGUILLIER']['password']
                            ),
                        params={'customer': self.name}
                        )
        return cdrs_by_policies

    def get_cdr_summary(self, start=None, stop=None):
        cdrs = self.get_cdr(start, stop)
        sorted_cdrs = sorted(
            cdrs,
            key=lambda x: datetime.strptime(x, '%d%m%Y')
            )
        infos = {
            'refnum': self.refnum,
            'name': self.name,
            'consumption': {
                'start': start.strftime('%d%m%Y'),
                'stop': stop.strftime('%d%m%Y'),
                'length': 0,
                'details': []
                }
            }
        for date in sorted_cdrs:
            data = cdrs[date]
            octets_in = sum([i['octets_in'] for i in data])
            octets_out = sum([i['octets_out'] for i in data])
            info = {
                'date': date,
                'total': octets_in + octets_out,
                'octets_in': octets_in,
                'octets_out': octets_out,
                'policy_wise': []
                }
            cdrs_by_policies = self.filter_cdrs_policy_wise({date: data})
            for i in cdrs_by_policies:
                del i['details']
                info['policy_wise'].append(i)
            infos['consumption']['details'].append(info)
            infos['consumption']['length'] += 1
        LOGS.logger.debug(infos)
        return infos

# EOF
