#!/usr/bin/env python
# encoding: utf-8

from datetime import datetime, timedelta
from email.mime.multipart import MIMEMultipart
from email.mime.base import MIMEBase
from email.mime.text import MIMEText
from multiprocessing import Pool

from .. import core
from ..consts import CONFIG, REDIS_DB
from ..customer import Customer
from ..product import Product
from ..product_gp import Product_GP
from . import app

import click
import smtplib
import tablib


@click.group()
def cmd():
    pass


@cmd.command()
@click.option('--host', default='0.0.0.0')
@click.option('--port', default=5000)
def runserver(host, port):
    """Lancer le serveur Werkzeug pour l'API REST (dev. seulement)"""
    app.run(host=host, port=port, debug=True)


@cmd.command()
def sync():
    """Mise à jour de la liste des clients Business ainsi que de leurs \
    relevés de consommation des 30 derniers jours."""
    products = Product.all()
    for product in products:
        product.update_customers()
        with Pool(processes=2) as pool:
            pool.map(core.update_customer_cdr, product.customers)


@cmd.command()
def notify():
    """Commande pour notifier les clients grand public \
    selon leur niveau de consommation"""
    click.echo('Récupération des données depuis Bluebase')
    customers = Product_GP.all()
    for customer in customers:
        click.echo('Notification du client {}'.format(customer.msisdn))
        response = customer.notify()
        if not response:
            click.echo("Pas de notification à envoyer")
        else:
            medium, percentage, status = response
            msg = '{}: {}%'.format(medium, round(percentage))
            if status:
                msg = '{} ({})'.format(msg, 'Notification envoyée')
            click.echo(msg)


@cmd.command()
@click.option('--append-details', is_flag=True, default=False)
@click.option('--for-real', is_flag=True, default=False)
@click.option('--send-mail', is_flag=True, default=False)
def apply(append_details, for_real, send_mail):
    """Appliquer les politiques de FUP pour tous les clients Business"""
    result = {}
    products = Product.all()
    for product in products:
        policies = product.apply_policies(for_real=for_real)
        if policies:
            result[product.name] = policies
    if result:
        message = ''
        dataset = tablib.Dataset()
        dataset.headers = [
            'product',
            'data_limit',
            'policy_name',
            'customer',
            'download',
            'upload',
            'total',
            'status'
            ]
        products = sorted([i for i in result], reverse=True)
        for product in products:
            policies = result[product]
            message += 'Produit: {product}\n'.format(product=product)
            for policy in policies:
                message += (
                    '\n'
                    'Limite autorisée: {data_limit} Go\n'
                    ).format(
                        data_limit=policy['data_limit']
                        )
                foo = ''
                for customer, data in policy['customers'].items():
                    customer_policies_status = Customer(customer).fup_status
                    policy_status = False
                    customer_policy_status = customer_policies_status.get(
                        policy['name']
                        )
                    if customer_policy_status:
                        customer_policy_status = sorted(
                            customer_policy_status,
                            key=lambda x: x['datetime'],
                            reverse=True
                            )
                        policy_status = customer_policy_status[0]['status']
                    dataset.append(
                        [
                            product,
                            policy['data_limit'],
                            policy['name'],
                            customer,
                            data['octets_in'],
                            data['octets_out'],
                            data['total'],
                            policy_status
                            ]
                        )
                    if data.get('fup_in'):
                        status = 'fup_in'
                    elif data.get('fup_out'):
                        status = 'fup_out'
                    else:
                        status = None
                    if status:
                        foo += (
                            ' - {customer}: {total} Go ({status})\n'.format(
                                customer=customer,
                                total=data['total'],
                                status=status
                                )
                            )
                if foo:
                    message += foo
                else:
                    message += "Aucun client à FUPer\n"
            message += '\n\n\n'
        click.echo(message)
        with open('/tmp/details.csv', 'w') as f:
            f.write(dataset.export('csv'))
        if send_mail:
            recipients = CONFIG['ALERT']['to']
            content = MIMEText(message, 'plain', 'utf-8')
            attachment = MIMEBase('text', 'csv')
            fp = open('/tmp/details.csv', 'rb')
            attachment.set_payload(fp.read())
            fp.close()
            attachment.add_header(
                'Content-Disposition',
                'attachment',
                filename='details.csv'
                )
            msg = MIMEMultipart()
            msg['Subject'] = CONFIG['ALERT']['subject']
            msg['From'] = CONFIG['ALERT']['from']
            msg['To'] = recipients
            msg.attach(attachment)
            msg.attach(content)
            s = smtplib.SMTP('localhost')
            s.sendmail(
                CONFIG['ALERT']['from'],
                recipients.replace(' ', '').split(','),
                msg.as_string()
                )
            s.quit()


@cmd.command()
@click.option('--start')
@click.option('--stop')
@click.option('--send-mail')
def details(start, stop, send_mail):
    """Renvoie le détail de consommation des clients"""
    if not stop:
        stop = datetime.now() - timedelta(1)
    if not start:
        start = datetime.now() - timedelta(30)
    dataset = tablib.Dataset()
    dataset.headers = ['client', 'produit', 'politique_fup', 'total']
    date = start
    while date <= stop:
        dataset.headers.append(date.strftime('%d%m%Y'))
        date += timedelta(1)
    customers = Customer.all()
    cdrs = {}
    with click.progressbar(customers) as _customers:
        for customer in _customers:
            key = 'cdr_start_date:{}'.format(customer.name)
            _start = REDIS_DB.get(key)  # s'il n'y a pas de clés ce sera None
            if _start:
                _start = _start.decode('utf-8')
                start = datetime.strptime(
                    _start + ' 00:00:00', '%d%m%Y %H:%M:%S'
                    )
            msg = '> Conso de {} du {} au {}'.format(
                customer.name,
                start.strftime('%d/%m/%Y'),
                stop.strftime('%d/%m/%Y')
                )
            click.echo(msg)
            cdrs[customer.name] = {
                'product': customer.product.name,
                'cdrs': customer.filter_cdrs_policy_wise(
                    customer.get_cdr(start=start, stop=stop)
                    )
                }
    for customer, infos in cdrs.items():
        product = infos['product']
        data = infos['cdrs']
        for policy_cdrs in data:
            policy_name = policy_cdrs['name']
            total = round(
                policy_cdrs['total']/1000000000, 2
                )
            line = [customer, product, policy_name, total]
            for date in dataset.headers[4:]:
                for cdr in policy_cdrs['details']:
                    if cdr['date'] == date:
                        line.append(
                            round(
                                cdr['total']/1000000000, 2
                                )
                            )
                        break
                else:
                    line.append(0)
            dataset.rpush(line)
    with open('/tmp/conso.csv', 'w') as f:
        f.write(dataset.export('csv'))
    if send_mail:
        message = "Consommation des clients du {} au {}".format(
            start.strftime('%d%m%Y'),
            stop.strftime('%d%m%Y')
            )
        content = MIMEText(message, 'plain', 'utf-8')
        attachment = MIMEBase('text', 'csv')
        fp = open('/tmp/conso.csv', 'rb')
        attachment.set_payload(fp.read())
        fp.close()
        attachment.add_header(
            'Content-Disposition',
            'attachment',
            filename='conso.csv'
            )
        msg = MIMEMultipart()
        msg['Subject'] = "Détails de consommation FUP"
        msg['From'] = CONFIG['ALERT']['from']
        msg['To'] = send_mail
        msg.attach(attachment)
        msg.attach(content)
        s = smtplib.SMTP('localhost')
        s.sendmail(
            CONFIG['ALERT']['from'],
            send_mail.replace(' ', '').split(','),
            msg.as_string()
            )
        s.quit()
# EOF
