#!/usr/bin/env python
# encoding: utf-8

"""
Ce module contient les briques de bases du service
de gestion des politiques de consommation
des utilisateurs Internet chez Blueline.

Une synchronisation des données est effectuée tous les matins
avec Aiguillier pour disposer d'une base locale afin de faire
des calculs sans craindre les ralentissements réseau. Toutes
les vérifications se font donc en local.
"""
import smtplib
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.utils import formatdate

from .consts import FUPDB_CREDENTIALS, APIGW, LOGS

import records
import psycopg2


__all__ = [
    'database_url',
    'record_customer',
    'apply_product_policies',
    'update_customer_cdr'
    ]


def connexion(**kwargs):
    con = psycopg2.connect(
        user=kwargs.get('user'),
        password=kwargs.get('password'),
        host=kwargs.get('host'),
        port=kwargs.get('port'),
        database=kwargs.get('name')
    )
    return con


def database_url(**kwargs):
    url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
        eng=kwargs.get('engine'),
        user=kwargs.get('user'),
        password=kwargs.get('password'),
        host=kwargs.get('host'),
        port=kwargs.get('port'),
        name=kwargs.get('name')
        )
    return url


def purge_customer(name):
    url = database_url(**FUPDB_CREDENTIALS)
    with records.Database(url) as db:
        sql = "DELETE FROM customer WHERE name=:name"
        db.query(sql, name=name)
        sql = "DELETE FROM cdr WHERE customer=:name"
        db.query(sql, name=name)
        sql = "DELETE FROM fup WHERE customer=:name"
        db.query(sql, name=name)


def record_customer(infos):
    url = database_url(**FUPDB_CREDENTIALS)
    with records.Database(url) as db:
        sql = (
            "INSERT INTO customer (name, refnum, product) "
            "VALUES (:name, :refnum, :product)"
            )
        db.query(
            sql,
            name=infos['name'],
            refnum=infos['refnum'],
            product=infos['product']
            )


def apply_product_policies(for_real, customer):
    """Sous routine appliquant les politiques de bon usage
    à un client en particulier.
    Utile pour ensuite faire de l'exécution parallèle"""
    response = customer.apply_product_policies(for_real=for_real)
    data = [
        policy for policy in response
        # if 'fup_in' in policy or 'fup_out' in policy
        ]
    return customer, data


def update_customer_cdr(last_date, customer):
    """Mise à jour des CDRs d'un client dans la base REDIS"""
    customer.get_cdr_from_aiguillier(
        last_date=last_date
        )


def send_error_report(operation, error):
    """
    Sends an email to development and monitoring team to notify an error
    in the process.
    """
    message = (
        "Service: {} [{}]\n\n"
        "Affected operation: {}\n\n"
        "Error: {}"
        .format("fup-services", APIGW, operation, error)
    )
    sender = "api-gateway@si.blueline.mg"
    destinations = ["dev@si.blueline.mg", "sysadmin@si.blueline.mg"]
    content = MIMEText(message, 'plain', 'utf-8')

    msg = MIMEMultipart()
    msg['Subject'] = "fup-services process error"
    msg['From'] = sender
    msg['To'] = ', '.join(destinations)
    msg['Date'] = formatdate(localtime=True)
    msg.attach(content)
    smtp = smtplib.SMTP('localhost')
    smtp.sendmail(sender, destinations, msg.as_string())
    smtp.quit()

    LOGS.logger.info(
        "Email: '{}' sent to {}".format(msg['Subject'], destinations))
    smtp.quit()

# EOF
