# -*- coding: utf-8 -*-

from datetime import datetime, timedelta
from itertools import groupby

from .consts import CONFIG
from .consts import FUPDB_CREDENTIALS
from .consts import LOGS
from .consts import REDIS_DB
from .consts import REST_CLIENT
from .consts import URL_V2
from . import core
from . import product

import records
import json
import requests

__all__ = [
    'Customer',
    ]


class Customer(object):
    """Un client internet est identifié par un nom, un refnum et un produit"""

    def __init__(self, name, product_ese=False, **kwargs):
        self.name = name.lower()
        # url = core.database_url(**FUPDB_CREDENTIALS)
        # with records.Database(url) as db:
        #     if kwargs:
        #         self.table_customer = kwargs.get('table_customer')
        #         self.table_cdr = kwargs.get('table_cdr')
        #     else:
        #         self.table_customer = 'customer'
        #         self.table_cdr = 'cdr'
        #     sql = (
        #         "SELECT refnum, product FROM {}"
        #         " WHERE name=:name"
        #         ).format(self.table_customer)
        #     rows = db.query(sql, name=self.name)
        #     row = rows.first()
        #     if not row:
        #         raise ValueError(
        #             "Le client {name} n'existe pas".format(name=name)
        #             )
        #     else:
        #         self.product = product.Product(row.product)
        #         self.refnum = row.refnum
        #     LOGS.logger.info(self.product)
        if kwargs:
            self.table_customer = kwargs.get('table_customer')
            self.table_cdr = kwargs.get('table_cdr')
        else:
            self.table_customer = 'customer'
            self.table_cdr = 'cdr'
        conn = core.connexion(**FUPDB_CREDENTIALS)
        cursor = conn.cursor()
        sql = (
            "select refnum, product"
            " from {} where name = %s"
            ).format(self.table_customer)
        cursor.execute(sql, (self.name,))
        row = cursor.fetchone()
        LOGS.logger.info(
            "Customer {} information from CDR database: {}"
            .format(self.name, row)
        )
        if not row:
            raise ValueError(
                "Le client {name} n'existe pas".format(name=name)
                )
        else:
            self.product = product.Product(row[1])
            self.refnum = row[0]

    def __repr__(self):
        return '<Customer {name}>'.format(name=self.name)

    @staticmethod
    def all():
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = "SELECT name FROM customer"
            rows = db.query(sql)
            customers = [Customer(row.name) for row in rows]
        LOGS.logger.info(
            "GP customers count fetched inside SI databse: {}"
            .format(len(customers))
        )
        return customers

    @staticmethod
    def all_ese():
        kwargs = {
            'table_customer': 'customer_ese',
            'table_cdr': 'cdr_ese'
            }
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = "SELECT name FROM customer_ese"
            rows = db.query(sql)
            customers = [Customer(row.name, **kwargs) for row in rows]
        LOGS.logger.info(
            "ESE customers count fetched inside SI databse: {}"
            .format(len(customers))
        )
        return customers

    def customers_fup(self):
        """get info client fupé|défupé"""
        infos = {}
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = (
                "SELECT policy,datetime,customer, status, conso FROM fup "
                "WHERE customer=:customer and datetime=(SELECT max(datetime)"
                " from fup WHERE customer=:customer)"
                )
            rows = db.query(
                sql,
                customer=self.name,
                )
            for row in rows:
                infos = {
                    'customer': row.customer,
                    'status': row.status,
                    'conso': row.conso,
                    'policy': row.policy
                    }
                LOGS.logger.info(
                    "Customer '{}' status in SI database: {}"
                    .format(self.name, infos)
                )
        return infos

    @property
    def fup_status(self):
        """Retourne le status FUP du client"""
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = "SELECT * FROM fup WHERE customer=:name"
            rows = db.query(sql, name=self.name)
            result = {}
            for row in rows:
                if row.policy not in result:
                    result[row.policy] = []
                result[row.policy].append(
                    {
                        'datetime': row.datetime,
                        'status': row.status
                        }
                    )
            LOGS.logger.info(
                "Customer '{}' FUP history in SI database: {}"
                .format(self.name, result)
            )
        return result

    def set_refnum(self, refnum):
        """Mise à jour du refnum du client dans la base SI"""
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = (
                "UPDATE {} SET refnum=:refnum "
                "WHERE name=:name"
                ).format(self.table_customer)
            LOGS.logger.info(sql)
            db.query(
                sql,
                refnum=refnum,
                name=self.name
                )
        LOGS.logger.info(
            "Customer '{}' reference number updated from {} to {}"
            .format(self.name, self.refnum, refnum)
        )
        self.refnum = refnum

    def set_product(self, product_name):
        """Mise à jour du produit du client dans la base SI"""
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = (
                "UPDATE {} SET product=:product "
                "WHERE name=:name"
                ).format(self.table_customer)
            LOGS.logger.info(sql)
            db.query(
                sql,
                product=product_name,
                name=self.name
                )
        LOGS.logger.info(
            "Customer '{}' product updated from {} to {}"
            .format(self.name, self.product, product_name)
        )
        self.product = product.Product(product_name)

    def start_consumptions(self, start_conso):
        """
        enregistre dans redis la date a partir de
        laquelle on calcul la conso fup d'un client
        """
        result = {}
        key = 'cdr_start_date:{}'.format(self.name)
        time_expire = datetime.strptime(
            start_conso,
            '%d%m%Y'
            ) + timedelta(30) - datetime.strptime(
            datetime.now().strftime('%d%m%Y'),
            '%d%m%Y'
            )
        REDIS_DB.set(key, start_conso)
        REDIS_DB.expire(
            key,
            int(time_expire.total_seconds())
            )
        start = REDIS_DB.get(key)
        result = {
                'code': '200',
                'ident': self.name,
                'date': str(start.decode('utf-8')),
                'expire': str(time_expire)
                }
        LOGS.logger.info(
            "Clé Redis de la transaction: {}, début de la consommation: {}, "
            "date d'expiration{}"
            .format(key, start_conso, time_expire)
        )
        return result

    def fup_defup(
            self,
            method,
            params,
            policy_name,
            db_status,
            conso,
            period,
            msg
            ):
        api_request = getattr(REST_CLIENT, method)
        try:
            response = api_request(
                '/fup',
                auth=(
                    CONFIG['AIGUILLIER']['username'],
                    CONFIG['AIGUILLIER']['password']
                    ),
                params=params
                )
            LOGS.logger.info(
                "[FUP/DEFUP] Customer {} {operation} from Aiguiller: "
                "api request: {}, "
                "parameters: {}, "
                "response: {} {}"
                .format(
                    self.name,
                    api_request,
                    params,
                    response,
                    response.text,
                    operation=msg,
                )
            )
        except Exception as e:
            LOGS.logger.error(
                "Unable to {} customer {} from Aiguillier: {}"
                .format(msg, self.name, str(e))
            )
            core.send_error_report(
                "{} Customer from Aiguillier".format(msg),
                str(e)
            )
        LOGS.logger.info("{} un client {}-{}-{}-{}".format(
            msg,
            self.name,
            policy_name,
            period,
            response['status']
            ))
        if (response['status']).lower() == '200':
            url = core.database_url(**FUPDB_CREDENTIALS)
            with records.Database(url) as db:
                sql = (
                    "INSERT INTO fup "
                    "(customer, product, policy, datetime, status, "
                    "conso) "
                    "VALUES "
                    "(:name, :product, :policy, :date_time, :status, "
                    ":conso)"
                    )
                db.query(
                    sql,
                    name=self.name,
                    product=self.product.name,
                    policy=policy_name,
                    date_time=datetime.now(),
                    status=db_status,
                    conso=str(conso)
                    )
            LOGS.logger.info(
                "[FUP/DEFUP] Customer {} added to SI FUP database."
                .format(self.name)
            )
            result = {
                'code': '200',
                'message': '{} reussit'.format(msg)
                }
        else:
            result = {
                'code': '400',
                'message': 'ERROR {} COTE AIGUILLIER'.format(msg)
                }
        return result

    def defup_from_aiguillier(self, bool_defup):
        """

        :bool_defup:
        """
        result = {
            'code': '401',
            'message': 'NOK'
            }
        bool_delete = False
        params = {}
        params['customer'] = self.name
        if bool_defup:
            method = 'delete'
            msg = 'Defup'
            db_status = False
        else:
            method = 'post'
            msg = 'Fup'
            db_status = True
        customer_status = self.fup_status
        for policy_name in self.product.policies:
            status = customer_status.get(policy_name)
            policy_name_tab = policy_name.rsplit('_', 1)
            if policy_name_tab[1] == '2':
                period = 'hc'
                params['period'] = period
            elif policy_name_tab[1] == '3':
                period = 'try'
                method = 'post'
                if bool_defup:
                    params['actif'] = 'TRUE'
                    msg = 'Active'
                else:
                    msg = 'Desactive'
                    params['actif'] = 'FALSE'
            else:
                period = 'hp'
                params['period'] = period
            if status:
                status = sorted(
                    status,
                    key=lambda x: x['datetime'],
                    reverse=True
                    )
                if status[0]['status'] == bool_defup:
                    result = self.fup_defup(
                        method,
                        params,
                        policy_name,
                        db_status,
                        '0.0',
                        period,
                        msg
                        )
                    if result['code'] == '200':
                        bool_delete = True
                else:
                    result = {
                            'code': '201',
                            'message': 'deja {}'.format(msg)
                            }
            else:
                if not status and not bool_defup:
                    result = self.fup_defup(
                        method,
                        params,
                        policy_name,
                        db_status,
                        '0.0',
                        period,
                        msg
                        )
                    if result['code'] == '200':
                        bool_delete = True
                else:
                    result = {
                            'code': '204',
                            'message': 'produit non {}'.format(msg)
                            }
        # defup aiguiller v2
        if bool_defup:
            res = requests.delete('{}={}&periode={}'.format(
                URL_V2,
                self.name,
                period
                ),
                auth=(
                    CONFIG['AIGUILLIER']['username'],
                    CONFIG['AIGUILLIER']['password']
                    )
                )
            if res.status_code == 200:
                res = res.json()
                if res['status'] == '200':
                    LOGS.logger.info(
                        "[DEFUP] Customer {} removed from DT database"
                        .format(self.name)
                    )
                else:
                    LOGS.logger.error(
                        "[DEFUP] Unable to remove customer {} from DT database"
                        .format(self.name)
                    )
            else:
                LOGS.logger.error(
                    "[DEFUP] Unable to remove customer {} from DT database"
                    .format(self.name)
                )

        if bool_delete and bool_defup:
            url = core.database_url(**FUPDB_CREDENTIALS)
            with records.Database(url) as db:
                sql = "DELETE FROM customer WHERE name=:name"
                db.query(sql, name=self.name)
            LOGS.logger.info(
                "[DEFUP] Customer {} removed from SI FUP database"
                .format(self.name)
            )
        return result

    def get_cdr_from_aiguillier(self, last_date):
        """Récupération des CDRs a une date donné
        depuis l'API d'Aiguillier pour les clients"""

        # ====================================================================
        #  TODO: fetch CDRs directly from BI database
        # url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
        #     eng=CONFIG['BI']['engine'],
        #     user=CONFIG['BI']['user'],
        #     password=CONFIG['BI']['password'],
        #     host=CONFIG['BI']['host'],
        #     port=CONFIG['BI']['port'],
        #     name=CONFIG['BI']['table_name']
        # )
        # date = last_date + " 00:00:00"
        # sql = (
        #     "SELECT * FROM all_cdr WHERE "
        #     "clientname='{}' AND "
        #     "sessiontime >= '{}'::date"
        #     .format(self.name, last_date)
        # )
        # try:
        #     with records.Database(url) as db:
        #         rows = db.query(sql)
        #         cdrs_raw = rows.as_dict()
        #     LOGS.logger.info(
        #         "{} CRDs fetched from BI database for customer {}"
        #         .format(len(cdrs_raw), self.name)
        #     )
        # except Exception as e:
        #     LOGS.logger.error(
        #         "Unable to fetch customer {} CDRs: {}"
        #         .format(self.name, str(e))
        #     )
        #     core.send_error_report("CDRs synchronization", str(e))
        #     # we don't raise an exception, we assume customer has not used
        #     # their Internet
        #
        # cdrs = sorted(
        #     cdrs_raw or [],
        #     key=lambda x: datetime.strptime(
        #         x['sessiontime'],
        #         '%Y-%m-%dT%H:%M:%S'
        #     )
        # )
        # ====================================================================

        # =================== fetching CRDs via Aiguiller ====================
        import time
        url = core.database_url(**FUPDB_CREDENTIALS)

        OK = False
        for i in range(3):
            if OK:
                break
            try:
                LOGS.logger.info(
                    "[customer {}] Fetching {} CDRs try {}/3"
                    .format(self.name, last_date, i+1)
                )
                response = requests.get(
                    CONFIG['AIGUILLIER']['url_v2'] + "/consommation",
                    auth=(
                        CONFIG['AIGUILLIER']['username'],
                        CONFIG['AIGUILLIER']['password']
                    ),
                    params={"client": self.name, "depuis_date": last_date}
                )
                infos = response.json()
                LOGS.logger.info(
                    '[customer {}] CDRs count fetched from Aiguiller: {}'
                    .format(self.name, response)
                )
                if infos['status']:
                    OK = True
            except requests.exceptions.Timeout:
                LOGS.logger.info("essai TimeoutError {} de {}".format(
                    i,
                    self.name
                ))
                time.sleep(60)
                continue
            except ValueError:
                LOGS.logger.info("essai ValueError {} de {}".format(
                    i,
                    self.name
                ))
                time.sleep(60)
                continue
            except Exception as e:
                LOGS.logger.error(
                    "Unable to fetch customer {} CDRs: {}"
                    .format(self.name, str(e))
                )
                core.send_error_report("CDRs synchronization", str(e))
                # we don't raise an exception, we assume customer has not used
                # their Internet

        if type(infos) is str:
            infos = json.loads(infos)
        cdrs = sorted(
            infos.get('cdr', []),
            key=lambda x: datetime.strptime(
                x['sessiontime'],
                '%Y-%m-%dT%H:%M:%S'
            )
        )
        # ====================================================================

        cdrs = groupby(
            cdrs,
            key=lambda x: datetime.strptime(
                x['sessiontime'].split('T')[0],
                '%Y-%m-%d'
                )
            )
        result = {}
        params = []
        for date, infos in cdrs:
            LOGS.logger.info(
                "{} {} obtenu depuis Aiguillier".format(
                    self.name,
                    date.strftime('%d%m%Y')
                    )
                )
            result[date.strftime('%d%m%Y')] = []
            for info in infos:
                date_time = datetime.strptime(
                    info['sessiontime'],
                    '%Y-%m-%dT%H:%M:%S'
                    )
                params.append(
                    {
                        'name': self.name,
                        'product': self.product.name,
                        'date_time': date_time,
                        'source': info['source'],
                        'octetsin': info.get('octetsin', 0),
                        'octetsout': info.get('octetsout', 0)
                        }
                    )
                time = info['sessiontime'].split('T')[1].replace(':', '')
                values = {
                    'octets_in': info['octetsin'],
                    'octets_out': info['octetsout']
                    }
                values['time'] = datetime.strptime(time, '%H%M%S')
                values['source'] = info['source']
                values['date'] = date
                result[date.strftime('%d%m%Y')].append(values)
            with records.Database(url) as db:
                sql = (
                    "DELETE FROM {} WHERE customer=:name "
                    "AND datetime>=:start AND datetime<=:stop"
                    ).format(self.table_cdr)
                # start_date = date.strftime('%Y%m%d')+last_date[-6:]
                start_date = datetime.strptime(
                    str(last_date),
                    '%Y%m%d%H%M%S'
                    )
                LOGS.logger.info('start {} stop {}'.format(
                    start_date.strftime('%Y-%m-%d %H:%M:%S'),
                    date.strftime('%Y-%m-%d 23:59:59')
                    )
                )
                db.query(
                    sql,
                    name=self.name,
                    start=start_date.strftime('%Y-%m-%d %H:%M:%S'),
                    stop=date.strftime('%Y-%m-%d 23:59:59')
                    )
        if params:
            with records.Database(url) as db:
                sql = (
                    "INSERT INTO {} "
                    "(customer, datetime, source, "
                    "octets_in, octets_out, product) "
                    "VALUES "
                    "(:name, :date_time, :source, "
                    ":octetsin, :octetsout, :product)"
                    ).format(self.table_cdr)
                db.bulk_query(sql, *params)
                LOGS.logger.info(
                    "Parameters inserted inside SI database: {}"
                    .format(params)
                )
        return result

    def get_cdr(self, start=None, stop=None, product_out_fup=False):
        """Récupération des données de conso d'un client business.
        Les dates sont des objets datetime.datetime
        """
        if product_out_fup:
            table_cdr = 'cdr_ese'
        else:
            table_cdr = 'cdr'
        if not start:
            start = (datetime.now() - timedelta(30))
        if not stop:
            stop = (datetime.now() - timedelta(1))
        if stop < start:
            raise ValueError(
                "La date de fin est inférieure à la date de début"
                )
        LOGS.logger.info(
            "Recuperation des CDRS de {customer} "
            "du {start} au {stop} dans table {table} ...".format(
                customer=self.name,
                start=start.strftime('%d%m%Y'),
                stop=stop.strftime('%d%m%Y'),
                table=table_cdr
                )
            )
        cdrs = {}
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = (
                "SELECT * FROM {} "
                "WHERE customer=:name "
                "AND datetime >= :start "
                "AND datetime <= :stop "
                ).format(table_cdr)
            rows = db.query(
                sql,
                name=self.name,
                start='{} {}'.format(start.strftime('%Y-%m-%d'), '00:00:00'),
                stop='{} {}'.format(stop.strftime('%Y-%m-%d'), '23:59:59')
                )
            infos = [
                {
                    'date': row.datetime.date(),
                    'time': row.datetime.strftime('%H%M%S'),
                    'source': row.source,
                    'octets_in': row.octets_in,
                    'octets_out': row.octets_out
                    }
                for row in rows
                ]
            infos = sorted(
                infos,
                key=lambda x: x['date']
                )
            infos = groupby(infos, key=lambda x: x['date'])
            for date, data in infos:
                cdrs[date.strftime('%d%m%Y')] = list(data)
        return cdrs

    def filter_cdrs_policy_wise(self, cdrs, product_out_fup=False):
        """Suivant la politique de bon usage du produit réorganiser les cdrs"""
        result = []
        if product_out_fup:
            dict_policies = self.product.no_policies.items()
        else:
            dict_policies = self.product.policies.items()
        LOGS.logger.info(dict_policies)
        for policy_name, policy in dict_policies:
            sources = {}
            info = {
                'data_limit': policy['data_limit'],
                'interval': policy['interval'],
                'name': str(policy_name),
                'total': 0,
                'octets_in': 0,
                'octets_out': 0,
                'details': []
                }
            if 'name_type' in policy.keys():
                info['name_type'] = policy['name_type']
            intervals = []
            for interval in info['interval'].split('_'):
                start, stop, day_start, day_stop = interval.split(',')
                intervals.append(
                    {
                        'start': datetime.strptime(start, '%H%M%S'),
                        'stop': datetime.strptime(stop, '%H%M%S'),
                        'day_start': int(day_start),
                        'day_stop': int(day_stop)
                        }
                    )
            for day, infos in cdrs.items():
                data = []
                for interval in intervals:
                    for i in infos:
                        time = datetime.strptime(i['time'], '%H%M%S')
                        date_iso_weekday = i['date'].isoweekday()
                        start_time = interval['start']
                        stop_time = interval['stop']
                        days_range = range(
                            interval['day_start'],
                            interval['day_stop']+1
                            )
                        in_days = date_iso_weekday in days_range
                        if stop_time >= start_time:
                            in_times = (start_time <= time <= stop_time)
                        else:
                            in_times = (
                                start_time <= time or time <= stop_time
                                )
                        if (in_times and in_days):
                            data.append(i)
                data = sorted(data, key=lambda x: x['source'])
                day_octets_in = sum(
                    [
                        i['octets_in'] for i in data
                        if i['source'] != 'national'
                        ]
                    )
                day_octets_out = sum(
                    [
                        i['octets_out'] for i in data
                        if i['source'] != 'national'
                        ]
                    )
                day_total = day_octets_in + day_octets_out
                data = groupby(data, key=lambda x: x['source'])
                day_details = []
                for source, element in data:
                    if source not in sources:
                        sources[source] = {'octets_in': 0, 'octets_out': 0}
                    element = list(element)
                    source_day_octets_in = sum(
                        [i['octets_in'] for i in element]
                        )
                    source_day_octets_out = sum(
                        [i['octets_out'] for i in element]
                        )
                    source_day_total = (
                        source_day_octets_in + source_day_octets_out
                        )
                    sources[source]['octets_in'] += source_day_octets_in
                    sources[source]['octets_out'] += source_day_octets_out
                    day_details.append(
                        {
                            'source': source,
                            'octets_in': source_day_octets_in,
                            'octets_out': source_day_octets_out,
                            'total': source_day_total
                            }
                        )
                info['total'] += day_total
                info['octets_in'] += day_octets_in
                info['octets_out'] += day_octets_out
                info['details'].append(
                    {
                        'date': day,
                        'octets_in': day_octets_in,
                        'octets_out': day_octets_out,
                        'total': day_total,
                        'details': day_details
                        }
                    )
            policy_name_tab = policy_name.rsplit('_', 1)
            if policy_name_tab[1] == '2':
                info['period'] = 'hc'
            elif policy_name_tab[1] == '3':
                info['period'] = 'try'
            else:
                info['period'] = 'hp'
            info['sources'] = [
                {
                    'source': source,
                    'octets_in': bar['octets_in'],
                    'octets_out': bar['octets_out']
                    }
                for source, bar in sources.items()
                ]
            result.append(info)
        return result

    def apply_product_policies(self, for_real=False):
        """Application des politiques de bon usage du produit pour business"""
        # on vérifie si le client a une configuration particulière
        # pour la date à laquelle on doit commencer à compter
        # sa consommation
        params = {}
        key = 'cdr_start_date:{}'.format(self.name)
        start = REDIS_DB.get(key)  # s'il n'y a pas de clés ce sera None
        if start:
            start = start.decode('utf-8')
            start = datetime.strptime(start + ' 00:00:00', '%d%m%Y %H:%M:%S')
        cdrs = self.get_cdr(start=start)
        cdrs_by_policies = self.filter_cdrs_policy_wise(cdrs)
        for info in cdrs_by_policies:
            total = info['total']
            octets_in = info['octets_in']
            octets_out = info['octets_out']
            info['total'] = round(total / 1000000000, 2)
            info['octets_in'] = round(octets_in / 1000000000, 2)
            info['octets_out'] = round(octets_out / 1000000000, 2)
            url = core.database_url(**FUPDB_CREDENTIALS)
            with records.Database(url) as db:
                sql = (
                    "SELECT status FROM fup "
                    "WHERE customer=:name "
                    "AND policy=:policy "
                    "ORDER BY datetime DESC LIMIT 1"
                    )
                rows = db.query(sql, name=self.name, policy=info['name'])
                row = rows.first()
                if row:
                    status = row.status
                else:
                    status = False
                LOGS.logger.info(
                    "Customer {} fup status in SI database: {}"
                    .format(self.name, status)
                )
            if info['total'] >= info['data_limit'] and not status:
                info['fup_in'] = True
            elif info['total'] < info['data_limit'] and status:
                info['fup_out'] = True
            if for_real:
                # FUP the customer in Aiguiller
                if info.get('fup_in'):  # à limiter
                    LOGS.logger.info(
                        "> FUP-ing Customer '{}'...".format(self.name)
                    )
                    if info['period'] == 'try':
                        params['actif'] = 'FALSE'
                    else:
                        params['period'] = info['period']
                    params['customer'] = self.name
                    self.fup_defup(
                        'post',
                        params,
                        info['name'],
                        True,
                        str(info['total']),
                        str(info['period']),
                        'Fup'
                        )
                    """url = core.database_url(**FUPDB_CREDENTIALS)
                    with records.Database(url) as db:
                        sql = (
                            "INSERT INTO fup "
                            "(customer, product, policy, datetime, status, "
                            "conso) "
                            "VALUES "
                            "(:name, :product, :policy, :date_time, :status, "
                            ":conso)"
                            )
                        db.query(
                            sql,
                            name=self.name,
                            product=self.product.name,
                            policy=info['name'],
                            date_time=datetime.now(),
                            status=True,
                            conso=str(info['total'])
                            )
                    if info['period'] == 'try':
                        params['actif'] = FALSE
                    else:
                        params['period'] = info['period']
                    params['customer'] = self.name
                    REST_CLIENT.post(
                        '/fup',
                        auth=(
                            CONFIG['AIGUILLIER']['username'],
                            CONFIG['AIGUILLIER']['password']
                            ),
                        params=params
                        )
                    """
                # DEFUP the customer in Aiguiller
                elif info.get('fup_out'):  # libérer
                    LOGS.logger.info(
                        "> DEFUP-ing Customer '{}'...".format(self.name)
                    )
                    if info['period'] == 'try':
                        params['actif'] = 'TRUE'
                        method = 'post'
                    else:
                        params['period'] = info['period']
                        method = 'delete'
                        params['customer'] = self.name
                        self.fup_defup(
                            method,
                            params,
                            info['name'],
                            False,
                            str(info['total']),
                            str(info['period']),
                            'Defup'
                            )
                    """url = core.database_url(**FUPDB_CREDENTIALS)
                    with records.Database(url) as db:
                        sql = (
                            "INSERT INTO fup "
                            "(customer, product, policy, datetime, status, "
                            "conso) "
                            "VALUES "
                            "(:name, :product, :policy, :date_time, :status, "
                            ":conso)"
                            )
                        db.query(
                            sql,
                            name=self.name,
                            product=self.product.name,
                            policy=info['name'],
                            date_time=datetime.now(),
                            status=False,
                            conso=str(info['total'])
                            )
                    if info['period'] == 'try':
                        params['actif'] = TRUE
                    else:
                        params['period'] = info['period']
                    params['customer'] = self.name
                    REST_CLIENT.delete(
                        '/fup',
                        auth=(
                            CONFIG['AIGUILLIER']['username'],
                            CONFIG['AIGUILLIER']['password']
                            ),
                        params=params
                        )
                    """
        return cdrs_by_policies

    def get_cdr_summary(
            self,
            start=None,
            stop=None,
            daily_details=False,
            product_out_fup=None,
            bool_start=False
            ):
        sources = ['cache', 'national', 'internet']
        if not bool_start:
            key = 'cdr_start_date:{}'.format(self.name)
            start_date = REDIS_DB.get(key)
            if start_date:
                start = start_date.decode('utf-8')
                start = datetime.strptime(start, '%d%m%Y').date()
        if not start:
            start = (datetime.now() - timedelta(30))
        if not stop:
            stop = (datetime.now() - timedelta(1))
        LOGS.logger.info('start_date {} stop_date {}'.format(
            start,
            stop
            ))
        cdrs = self.get_cdr(start, stop, product_out_fup)
        sorted_cdrs = sorted(
            cdrs,
            key=lambda x: datetime.strptime(x, '%d%m%Y')
            )
        infos = {
            'refnum': self.refnum,
            'name': self.name,
            'consumption': {
                'start': start.strftime('%d%m%Y'),
                'stop': stop.strftime('%d%m%Y'),
                'length': 0,
                'details': []
                }
            }
        for date in sorted_cdrs:
            daily_data = cdrs[date]
            daily_data = sorted(daily_data, key=lambda x: x['source'])
            daily_octets_in = sum([i['octets_in'] for i in daily_data])
            daily_octets_out = sum([i['octets_out'] for i in daily_data])
            daily_total = daily_octets_in + daily_octets_out
            daily_data_by_source = groupby(
                sorted(daily_data, key=lambda x: x['source']),
                key=lambda x: x['source']
                )
            total_by_source = []
            for source, foo in daily_data_by_source:
                foo = list(foo)
                octets_in = sum([i['octets_in'] for i in foo])
                octets_out = sum([i['octets_out'] for i in foo])
                total_by_source.append(
                    {
                        'source': source,
                        'octets_in': octets_in,
                        'octets_out': octets_out,
                        'total': octets_in + octets_out
                        }
                    )
            details_sources = []
            if total_by_source:
                details_sources = [
                    x['source'] for x in total_by_source
                ]
            for x in sources:
                if x not in details_sources:
                    total_by_source.append(
                        {
                            'source': x,
                            'octets_in': 0.0,
                            'total': 0.0,
                            'octets_out': 0.0
                        }
                    )
            info = {
                'date': date,
                'total': daily_total,
                'octets_in': daily_octets_in,
                'octets_out': daily_octets_out,
                'details': total_by_source,
                'policy_wise': []
                }
            cdrs_by_policies = self.filter_cdrs_policy_wise(
                {date: daily_data},
                product_out_fup
                )
            for i in cdrs_by_policies:
                # comme c'est sur une seule journée
                #  - i['details'] est de longueur 1
                #  - i['sources'] et i['details'][0]['details'] sont pareils
                details = i['details'][0]['details']
                details_sources = []
                if details:
                    details_sources = [
                        x['source'] for x in details
                    ]
                del i['details']
                del i['sources']
                if daily_details:
                    for x in sources:
                        if x not in details_sources:
                            details.append(
                                {
                                    'source': x,
                                    'octets_in': 0.0,
                                    'total': 0.0,
                                    'octets_out': 0.0
                                }
                            )
                    i['details'] = details
                info['policy_wise'].append(i)
            infos['consumption']['details'].append(info)
            infos['consumption']['length'] += 1
        LOGS.logger.debug(infos)
        return infos

# EOF
