#!/usr/bin/env python
# encoding: utf-8


"""
Interface en ligne de commande pour la gestion
du contenu LDAP
"""

from blueflask.lib.amqp_worker import worker, Worker
from ..core import Email
from ..objects import Department, User
from . import app

import click
import json


@click.group()
def cmd():
    pass


@cmd.command()
@click.argument('queue')
@click.option('--workers', type=int, default=1)
def async_daemon(queue, workers):
    """Lance un ou plusieurs workers pour une file d'attente"""

    if not workers:
        click.echo("Il me faut au moins un worker")
    elif workers == 1:
        worker(queue, 'ldap')
    else:
        threads = [
            Worker('worker-{}'.format(i), queue, 'ldap')
            for i in range(workers)
            ]
        for thread in threads:
            thread.start()


@cmd.command()
@click.argument('object_type', type=click.Choice(['department', 'user']))
@click.argument('name')
def get(object_type, name):
    """Lire les informations d'un objet LDAP"""
    try:
        if object_type == 'department':
            obj = Department(name)
        elif object_type == 'user':
            obj = User(name)
    except IndexError:
        click.echo(
            click.style("{} n'existe pas".format(name), fg='red'),
            err=True
            )
    else:
        response = {}
        for key, value in obj.__dict__.items():
            if key == 'manager':
                value = '{name} ({identifier})'.format(
                    name=value.name,
                    identifier=value.uid
                    )
            if key != 'handler':
                response[key] = value
        if object_type == 'department':
            response['members'] = []
            for role in obj.roles:
                occupants = role.occupants or []
                for occupant in occupants:
                    bar = {
                        'name': occupant.name,
                        'identifier': occupant.uid
                        }
                    response['members'].append(bar)
        elif object_type == 'user':
            response['departement'] = obj.department.name
        click.echo(
            click.style(
                json.dumps(response, indent=4),
                fg='yellow'
                )
            )


@cmd.command()
@click.argument('object_type', type=click.Choice(['department', 'user']))
@click.option('--name')
def search(object_type, name):
    """Rechercher des elements LDA"""
    if object_type == 'department':
        cls = Department
    elif object_type == 'user':
        cls = User
    if not name:
        obj = cls.all()
    else:
        obj = cls.search(name)
    response = []
    for i in obj:
        bar = {}
        bar['name'] = i.name
        if object_type == 'user':
            bar['uid'] = i.uid
        response.append(bar)
    if not response:
        click.echo(
            click.style('Aucun resultat', fg='red'),
            err=True
            )
    else:
        click.echo(
            click.style(
                json.dumps(response, indent=4),
                fg='yellow'
                )
            )


@cmd.command()
@click.argument('username')
def reset(username):
    """Modification du mot de passe d'un utilisateur"""

    try:
        user = User(username)
    except ValueError:
        click.echo(
            click.style(
                "{username} n'existe pas".format(username=username),
                fg='red'
                ),
            err=True
            )
    else:
        new_password = user.reset()
        Email().reset(user, new_password)


@cmd.command()
@click.option('--host', default='127.0.0.1')
@click.option('--port', default=5000)
def runserver(host, port):
    """Lancement du serveur de dev de l'API REST"""

    app.run(host=host, port=port, debug=True)

# EOF
