#!/usr/bin/env python
# encoding: utf-8


from flask import g, jsonify, request

from blueflask.lib.errors import bad_request, forbidden, not_found
from blueflask.lib.errors import unauthorized
from blueflask.lib.infos import created, no_content

from ...objects_odoo import LdapHandler, EmailHandler
from ...consts import SERVICE_CODE, LOGS, CONFIG
from . import api

import ldap


@api.route("odoo/users", methods=["GET"])
def get_all_users():
    """Endpoint for getting all users information."""
    ldap_handler = LdapHandler()
    user_l = ldap_handler.search(
        "User",
    )
    return jsonify(user_l['users'])


@api.route("odoo/user", methods=["GET"])
def get_ldap_user():
    """Endpoint for getting a user information.
    Input:
        'employeeNumber' - a registration_number
        OR
        'email' - a user email
    """
    parameters = request.get_json()
    if parameters.get('email'):
        identifier = "uid"
        value = parameters['email']
    else:
        identifier = "employeeNumber"
        value = parameters['employeeNumber']

    ldap_handler = LdapHandler()
    user_exists = ldap_handler.search(
        "User",
        identifier=identifier,
        value=value
    )
    if user_exists:
        _, user_info = ldap_handler.parse_search(user_exists[0])
        manager_tree = user_info.get('manager')
        if manager_tree:
            manager_name = manager_tree.split(",")[0].split("=")[1]
            manager_b = ldap_handler.search("User", identifier="cn", value=manager_name)
            LOGS.logger.info(manager_b)
            if manager_b:
                manager = ldap_handler.parse_search(manager_b[0])
                LOGS.logger.info(manager)
                user_info['manager'] = manager
            else:
                user_info['manager'] = []
        else:
            user_info['manager'] = []
        return jsonify(user_info)
    else:
        return "User not found.", 404


@api.route("odoo/user", methods=["POST"])
def create_ldap_user():
    """Endpoint for creating a new user.
    Input:
        sn: user last name,
        givenName: user firstname,
        mail: user work email,
        ou: a department name,
        employeeNumber: the employee registration number,
        employeeType: a job position name inside the given department,
        mobile: the user personal phone number,
        telephoneNumber: the user work phone number,
        jpegPhoto: the user image (not working currently)
        manager:  an existing user ldap tree
    Output:

    """
    parameters = request.get_json()
    parameters.update({'uid': parameters['mail']})
    generic_tree = CONFIG.get("User", "dn")
    ldap_handler = LdapHandler()
    user_exists = ldap_handler.search(
        "User",
        identifier="employeeNumber",
        value=parameters['employeeNumber']
    )
    if user_exists:
        return str(
            "This user already exists. If you mean to update it, "
            "this is the wrong entry point."
        )
    department = ldap_handler.search(
        "Department",
        identifier="ou",
        value=parameters['ou']
    )
    if not department:
        return "Department not found", 404
    department_tree, _ = ldap_handler.parse_search(department[0])
    job_tree = "cn={},{}".format(parameters['employeeType'], department_tree)
    full_name = "{} {}".format(parameters['sn'], parameters['givenName'])
    user_tree = "cn={},{}".format(full_name, generic_tree)
    parameters_b = [(key, value.encode("utf-8")) for key, value in parameters.items()]
    ldap_handler.create("User", user_tree, user_info=parameters_b)
    ldap_handler.add_to("User", user_tree, job_tree)
    # password = ldap_handler.initialize_access(parameters['mail'], user_tree)
    # email_content = {
    #     'cn': parameters['givenName'],
    #     'username': parameters['mail'],
    #     'dn': "",
    #     'mail': parameters['mail'],
    # }
    # EmailHandler().create(email_content, password)
    return "User created."


@api.route("odoo/user", methods=["PUT"])
def update_ldap_user():
    """
    Endpoint for updating a user. If a new job title (employeeType) is provided,
    user is removed from his old job title list before being given a new one.
    Input required:
        'employeeNumber' - a registration_number
    Input optional:
        manager: an existing user ldap tree
        'mail' - an email
        'uid' - same as email
        'givenName' - a first name
        'sn' - a last name
        'cn' - at least 1 first name and 1 last name
        'employeeType' - a job title
    """
    parameters = request.get_json()
    ldap_handler = LdapHandler()
    user_exists = ldap_handler.search(
        "User",
        identifier="employeeNumber",
        value=parameters['employeeNumber']
    )
    if user_exists:
        user_tree, user_info = ldap_handler.parse_search(user_exists[0])
        if 'employeeType' in parameters:
            generic_tree = CONFIG.get("Department", "dn")
            try:
                old_department_tree = "ou={},{}".format(user_info['ou'], generic_tree)
                old_job_tree = "cn={},{}".format(user_info['employeeType'], old_department_tree)
                ldap_handler.remove_from("User", user_tree, old_job_tree)
            except ldap.NO_SUCH_ATTRIBUTE:
                LOGS.logger.warning(
                    "[update_ldap_user]\n"
                    "> Could not find user '{}' inside Department '{}' "
                    "at position '{}'. User was not deleted from job tree.\n"
                    .format(
                        user_info['sn'],
                        user_info['ou'],
                        user_info['employeeType']
                    )
                )
            department_tree = "ou={},{}".format(parameters['ou'], generic_tree)
            job_tree = "cn={},{}".format(parameters['employeeType'], department_tree)
            ldap_handler.add_to("User", user_tree, job_tree)
        ldap_handler.update("User", user_tree, parameters, overwrite=True)
        return "User updated."
    else:
        return "User not found."


@api.route("odoo/user", methods=["DELETE"])
def delete_ldap_user():
    """Endpoint for creating a new user.
    required parameters:
        'employeeNumber' - a registration number
        'ou' - a Department name
        'employeeType' - a job title in the given Department
    """
    parameters = request.get_json()
    ldap_handler = LdapHandler()
    user_exists = ldap_handler.search(
        "User",
        identifier="employeeNumber",
        value=parameters['employeeNumber']
    )
    if user_exists:
        user_tree, info = ldap_handler.parse_search(user_exists[0])
        department = ldap_handler.search(
            "Department",
            identifier="ou",
            value=parameters['ou']
        )
        department_tree, _ = ldap_handler.parse_search(department[0])
        job_tree = "cn={},{}".format(info['employeeType'], department_tree)
        ldap_handler.delete("User", user_tree, job_tree=job_tree)
        return "User deleted"
    else:
        return "User not found."


@api.route("odoo/user/reset", methods=["PUT"])
def reset_user_password():
    """
    Endpoint for resetting a ldap user password.
    required parameters:
        'employeeNumber' - a registration_number
    """
    parameters = request.get_json()
    ldap_handler = LdapHandler()
    user_exists = ldap_handler.search(
        "User",
        identifier="employeeNumber",
        value=parameters['employeeNumber']
    )
    if user_exists:
        user_tree, user_info = ldap_handler.parse_search(user_exists[0])
        password = ldap_handler.generate_password()
        ldap_handler.update("User", user_tree, {"userPassword": password}, overwrite=True)
        EmailHandler().reset(
            name=user_info['sn'],
            username=user_info['uid'],
            password=password,
            email=user_info['mail']
        )
        return "Password reset."
    else:
        return "User not found."


@api.route("odoo/search", methods=["GET"])
def search_object():
    parameters = request.get_json()
    # if set(parameters.keys()) != ('identifier', 'value', 'object'):
    #     LOGS.logger.error("Missing parameters:", parameters)
    #     return "Missing parameters", 400

    identifier = parameters['identifier']
    value = parameters['value']
    requested_object = parameters['object']

    ldap_handler = LdapHandler()
    try:
        object_exists = ldap_handler.search(
            requested_object,
            identifier=identifier,
            value=value
        )
    except ldap.NO_SUCH_ATTRIBUTE:
        LOGS.logger.error("Unknown request: {}".format(parameters))
        return "Unknown request", 400

    if object_exists:
        _, object_info = ldap_handler.parse_search(object_exists[0])

        if requested_object == "User":
            manager_tree = object_info.get('manager')
            if manager_tree:
                manager_name = manager_tree.split(",")[0].split("=")[1]
                manager_b = ldap_handler.search("User", identifier="cn",
                                                value=manager_name)
                LOGS.logger.info(manager_b)
                if manager_b:
                    manager = ldap_handler.parse_search(manager_b[0])
                    LOGS.logger.info(manager)
                    object_info['manager'] = manager
                else:
                    object_info['manager'] = []
            else:
                object_info['manager'] = []
        LOGS.logger.info("Returned object: {}".format(object_info))
        return jsonify(object_info)
    else:
        LOGS.logger.info("Object not found: {}".format(parameters))
        return "Object not found.", 404
# EOF
