# -*- coding: utf-8 -*-

from flask import request, jsonify

from blueflask.lib.errors import bad_request, internal_error
from blueflask.lib.infos import success

from ...consts import CONFIG, SERVICE_CODE
from ... import cash_recharge, card_recharge, subscribe, transfer_account
from . import api


@api.route("/services/vouchers", methods=["POST"])
def activate_voucher():
    """
    Ressource permettant d'acheter du crédit via une carte à gratter.
    ---
    tags:
      - Services
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: phonenumber
        in: body
        type: string
        required: true
        description: Numéro de téléphone abonné
      - name: code
        in: body
        type: number
        required: true
        description: Numéro de la carte à gratter
    responses:
      200:
        description: L'opération a réussi
      500:
        description: Erreur interne à l'API
    """
    data = request.get_json()
    if data and data.get("operator"):
        operator_id = data.get("operator")
        del data["operator"]
        user = operator_id
    else:
        user = request.authorization.username
    if not data:
        return bad_request(service_code=SERVICE_CODE)
    else:
        phonenumber = data["phonenumber"][-9:]
        cardpinnumber = data["code"]
        interface = card_recharge.CardRechargeMgr(user)
        code, result = interface.VoucherRecharge(phonenumber, cardpinnumber)
        return_code = "{service_code}-{code}".format(
            service_code=SERVICE_CODE, code=code
        )
        if code == "405000000":
            response = {
                "status": 200,
                "code": return_code,
                "info": "activation réussie",
                "message": {
                    "balance": result.NewBalance,
                    "phonenumber": data["phonenumber"],
                    "face_value": result.FaceValue,
                },
            }
            return jsonify(response)
        elif code == "102010690":
            return bad_request(
                message="format voucher incorrect",
                code=return_code,
                service_code=SERVICE_CODE,
            )
        elif code == "102010671":
            return bad_request(
                message="carte de recharge invalide",
                code=return_code,
                service_code=SERVICE_CODE,
            )

        else:
            return internal_error(service_code=SERVICE_CODE, code=return_code)


@api.route("/services/transfers", methods=["POST"])
def transfer_money():
    """
    Ressource permettant d'échanger de l'argent.
    ---
    tags:
      - Services
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: sender
        in: body
        type: string
        required: true
        description: Numéro de téléphone émetteur
      - name: receiver
        in: body
        type: string
        required: true
        description: Numéro de téléphone receveur
      - name: amount
        in: body
        type: number
        required: true
        description: Montant du transfert
      - name: recharge_rule
        in: body
        type: string
        required: true
      - name: infos
        in: body
        type: string
    responses:
      200:
        description: L'opération a réussi
      404:
        description: Impossible de trouver le numéro de téléphone dans l'OCS
      400:
        description: Numéro de téléphone incorrect
      500:
        description: Erreur interne à l'API
    """
    data = request.get_json()
    if data and data.get("operator"):
        operator_id = data.get("operator")
        del data["operator"]
        user = operator_id
    else:
        user = request.authorization.username
    if not data:
        return bad_request(service_code=SERVICE_CODE)
    else:
        sender = data["sender"][-9:]
        del data["sender"]
        receiver = data["receiver"][-9:]
        del data["receiver"]
        amount = int(data["amount"])
        del data["amount"]
        if not sender.startswith("39"):
            interface = cash_recharge.CashRechargeMgr(user)
            code, result = interface.Payment(sender, receiver, amount, **data)
        else:
            interface = transfer_account.TransferAccountMgr(user)
            code, result = interface.TransferAccount(sender, receiver, amount)
        return_code = "{service_code}-{code}".format(
            service_code=SERVICE_CODE, code=code
        )
        if code == "405000000":
            return success(service_code=SERVICE_CODE)
        elif code == "405914555":
            return bad_request(
                message="Solde insuffisant", code=return_code, service_code=SERVICE_CODE
            )
        elif code == "405914562":
            return bad_request(
                message="Montant de transfert insuffisant",
                code=return_code,
                service_code=SERVICE_CODE,
            )
        else:
            return internal_error(service_code=SERVICE_CODE, code=return_code)


@api.route("/services/offers", methods=["POST"])
def change_offer():
    """
    Ressource permettant d'activer une offre.
    ---
    tags:
      - Services
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: phonenumber
        in: body
        type: string
        required: true
        description: Numéro de téléphone abonné
      - name: offer_id
        in: body
        type: number
        required: true
        description: ID de l'offre
    responses:
      200:
        description: L'opération a réussi
      500:
        description: Erreur interne à l'API
    """
    data = request.get_json()
    if data and data.get("operator"):
        operator_id = data.get("operator")
        del data["operator"]
        user = operator_id
    else:
        user = request.authorization.username
    if not data:
        return bad_request(service_code=SERVICE_CODE)
    else:
        phonenumber = data["phonenumber"][-9:]
        offer_id = data["offer_id"]
        interface = subscribe.SubscribeMgr(user)
        code, result = interface.changeOptionalOffer(phonenumber, offer_id)
        return_code = "{service_code}-{code}".format(
            service_code=SERVICE_CODE, code=code
        )
        if code == "405000000":
            return success(service_code=SERVICE_CODE)
        elif code == "405914012":
            return bad_request(
                message="Solde insuffisant", code=return_code, service_code=SERVICE_CODE
            )
        else:
            return internal_error(service_code=SERVICE_CODE, code=return_code)


@api.route("/services/offers", methods=["GET"])
def get_offers():
    """
    Ressource permettant d'avoir la liste des offres.
    ---
    tags:
      - Services
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: la liste des offres
    """
    related_object_ids = [
        i for i in CONFIG.sections() if i.startswith("BALANCE_RELATEDOBJECTID")
    ]
    offers = [dict(CONFIG[name]) for name in related_object_ids]
    message = {"length": len(offers), "offers": offers}
    return success(service_code=SERVICE_CODE, message=message)


# EOF
