# -*- coding: utf-8 -*-

"""
Ce module propose une classe permettant d'appeler une ressource
REST via HTTP sans se soucier de la gestion des erreurs et exceptions communes.
"""

import uuid

import requests


__all__ = [
    'RestClient'
    ]


class RestClient(object):

    def __init__(self, root_url, interface=None):
        """Création d'un nouveau client REST"""
        self.root_url = root_url
        self.interface = interface or requests

    def __repr__(self):
        return "<RestClient for {}>".format(self.root_url)

    def __call_url__(self, method, resource, **kwargs):
        """Méthode qui appelle effectivement une ressource.
        La ressource doit toujours être précédée d'un '/'.
        On peut rajouter des arguments nommées qui sont
        exactement les mêmes que ceux acceptés par la librairie 'requests'."""
        request = getattr(self.interface, method.lower())
        result = {  # le résultat de cette méthode sera toujours un <dict>
            'headers': None,
            'status': None,
            'content': None
            }
        url = '{root_url}{resource}'.format(
            root_url=self.root_url,
            resource=resource
            )
        kwargs = kwargs or {}
        headers = {  # si aucune en-tête n'est passée en paramètre, on en créé
            'X-Request-Id': kwargs.get(
                'headers', {}
                ).get(
                    'X-Request-Id',
                    str(uuid.uuid4())
                    )
            }
        auth = kwargs.get('auth', ())
        params = kwargs.get('params', None)  # les filtres de requêtes
        data = kwargs.get('data', None)  # les données de formulaires
        json = kwargs.get('json', None)  # les documents JSON
        try:
            response = request(
                url=url,
                headers=headers,
                auth=auth,
                params=params,
                data=data,
                json=json,
                timeout=None
                )
        except requests.exceptions.Timeout:
            msg = "{api} n'a pas répondu à temps (timeout).".format(api=url)
            result['headers'] = headers
            result['status'] = 504
            result['content'] = msg
        except requests.exceptions.ConnectionError:
            msg = "{api} n'est pas disponible".format(api=url)
            result['headers'] = headers
            result['status'] = 503
            result['content'] = msg
        else:
            try:
                content = response.json()
            except ValueError:
                content = response.text
            response.headers['X-Request-Id'] = headers.get('X-Request-Id')
            result['headers'] = response.headers
            result['status'] = response.status_code
            result['content'] = content
        return result  # <dict> contenant headers, status, content

    # Quelques raccourcis bien utiles pour gagner un peu de temps

    def get(self, resource, **kwargs):
        return self.__call_url__('get', resource, **kwargs)

    def post(self, resource, **kwargs):
        return self.__call_url__('post', resource, **kwargs)

    def put(self, resource, **kwargs):
        return self.__call_url__('put', resource, **kwargs)

    def delete(self, resource, **kwargs):
        return self.__call_url__('delete', resource, **kwargs)

    def head(self, resource, **kwargs):
        return self.__call_url__('head', resource, **kwargs)

    def patch(self, resource, **kwargs):
        return self.__call_url__('patch', resource, **kwargs)

    def options(self, resource, **kwargs):
        return self.__call_url__('options', resource, **kwargs)

# EOF
