#!/usr/bin/env python
# encoding: utf-8

from flask import request

from blueflask.lib.decorators import auth as ldap
from blueflask.lib.errors import bad_request
from blueflask.lib.infos import success

from ... import client
from .. import errors
from . import api


@api.route('/sms', methods=['POST'])
@ldap.login_required
def send_sms():
    """
    Envoi de SMS.
    Cette ressource contacte le SMSC pour envoyer des SMS
    ---
    tags:
      - sms
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: message
        in: body
        required: true
        description: Message court à envoyer
        schema:
          type: string
      - name: msisdn
        in: body
        required: true
        description: Numéro de téléphone destinataire
        schema:
          type: string
      - name: sender_id
        in: body
        required: false
        description: Identifiant SMPP à utiliser
        schema:
          type: string
      - name: simulation
        in: body
        required: false
        description: Interrupteur pour simuler ou non
        schema:
          type: string
          enum:
            - on
            - off
    responses:
      200:
        description: La demande d'envoi de SMS a été comprise par le SMSC
      400:
        description: Les paramètres en entrée sont incorrects
      503:
        description: Le SMSC met trop de temps à répondre
      504:
        description: Erreur de connection au SMSC
    """
    data = request.get_json()
    if 'message' not in data or 'msisdn' not in data:
        return bad_request()
    elif ',' in data['msisdn']:
        return errors.error(4000)
    elif not data['message']:
        return errors.error(4001)
    else:
        message = data['message']
        msisdn = data['msisdn']
        sender_id = data.get('sender_id', 'bip')
        encoding = data.get('encoding', 'DEFAULT')
        simulation = data.get('simulation', 'off')
        if simulation == 'off':
            simulate = False
        elif simulation == 'on':
            simulate = True
        else:
            return errors.error(4002)
        smpp_client = client.Client()
        try:
            smpp_client.send_sms(
                msisdn,
                message,
                simulate=simulate,
                request_id=request.headers['X-Request-Id'],
                sender_id=sender_id,
                encoding=encoding
                )
        except KeyError as exc:
            return errors.error(4003)
        if client.infos['submit']:
            return success("message soumis pour envoi au {}".format(msisdn))
        else:
            if client.infos['submit'] is None:
                return errors.error(5004)
            else:
                return errors.error(5003)

# EOF
