#!/usr/bin/env python
# encoding: utf-8

from datetime import datetime
from multiprocessing import Queue
from multiprocessing.queues import Empty

from systemd import journal

from .consts import CONFIG

import random

import smpplib.client
import smpplib.command
import smpplib.consts
import smpplib.exceptions
import smpplib.smpp
import smpplib.gsm


QUEUE = Queue()


class Client(object):

    def __init__(self, name='default'):
        self.msisdn = None
        smpp_config = 'SMPP_{}'.format(name)
        self.smpp_client = smpplib.client.Client(
            CONFIG[smpp_config]['host'],
            int(CONFIG[smpp_config]['port'])
            )
        self.smpp_client.set_message_sent_handler(self.callback)
        self.smpp_client.set_message_received_handler(self.callback)
        self.count = 0
        try:
            self.smpp_client.connect()
        except smpplib.exceptions.ConnectionError:
            raise SystemError("Impossible de se connecter au SMSC")
        else:
            self.smpp_client.bind_transceiver(
                system_id=CONFIG[smpp_config]['username'],
                password=CONFIG[smpp_config]['password']
                )
            from threading import Thread
            t = Thread(target=self.listen)
            t.start()

    def listen(self):
        while True:
            try:
                self.smpp_client.listen()
            except Exception as exc:
                print(self.msisdn, exc)

    def callback(self, pdu):
        """Si le SMSC dit qu'il a envoyé le SMS, on compte comme OK"""
        infos = {
            'date': datetime.now().strftime('%d-%m-%Y %H:%M:%S'),
            'msisdn': self.msisdn,
            'submit': True
            }
        msg = '<<< seq={} msg_id={} msisdn={}'.format(
            pdu.sequence,
            pdu.message_id,
            self.msisdn
            )
        journal.send(msg, REQUEST_ID=self.request_id)
        QUEUE.put(infos)

    def send_sms(self, msisdn, message, simulate=False, **kwargs):
        """Envoi du SMS au SMSC"""
        self.msisdn = msisdn
        self.sender_id = kwargs.get('sender_id', 'bip')
        try:
            source = CONFIG[self.sender_id]['source']
        except KeyError:
            source = CONFIG['bip']['source']
        self.request_id = kwargs['request_id']
        self.infos = {
            'msisdn': msisdn,
            'submit': False
            }
        if not simulate:
            if not kwargs.get('intl'):
                destination = '261{}'.format(msisdn[-9:])
            else:
                destination = msisdn
            parts, encoding_flag, msg_type_flag = smpplib.gsm.make_parts(
                message
                )
            # pour l'instant on laisse
            # SMPPLIB tenter du GSM-7, puis du UCS2
            for part in parts:
                pdu = self.smpp_client.send_message(  # NOQA
                    source_addr_ton=0,
                    source_addr_npi=0,
                    source_addr=source,
                    dest_addr_ton=0,
                    dest_addr_npi=0,
                    destination_addr=destination,
                    esm_class=msg_type_flag,
                    registered_delivery=1,
                    data_coding=encoding_flag,
                    short_message=part
                    )
                log_msg = '>>> seq={} msisdn={}'.format(
                    pdu.sequence,
                    self.msisdn
                    )
                journal.send(log_msg, REQUEST_ID=self.request_id)
        else:
            choice = random.randint(0, 9)
            if choice % 2:  # nombre impair
                infos = {
                    'date': datetime.now().strftime('%d-%m-%Y %H:%M:%S'),
                    'msisdn': msisdn,
                    'message_id': str(choice),
                    'submit': True
                    }
                QUEUE.put(infos)
        try:
            infos = QUEUE.get(timeout=1)
        except Empty:
            self.infos['date'] = datetime.now().strftime('%d-%m-%Y %H:%M:%S')
            self.infos['message'] = (
                "le SMSC n'a pas donné de réponse à temps"
                )
        else:
            self.infos = infos
            self.count += 1
        return

# EOF
