# -*- coding: utf-8 -*-

from flask import request

from blueflask.lib.errors import internal_error
from blueflask.lib.infos import success

from ...consts import CONFIG, SERVICE_CODE
from . import api

import json

import pika


@api.route('/sms', methods=['POST'])
def send_sms():
    """
    Envoi de SMS.
    Cette ressource contacte le SMSC pour envoyer des SMS
    ---
    tags:
      - sms
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: message
        in: body
        required: true
        description: Message court à envoyer
        schema:
          type: string
      - name: msisdn
        in: body
        required: true
        description: Numéro de téléphone destinataire
        schema:
          type: string
      - name: sender_id
        in: body
        required: false
        description: Identifiant SMPP à utiliser
        schema:
          type: string
      - name: simulation
        in: body
        required: false
        description: Interrupteur pour simuler ou non
        schema:
          type: string
          enum:
            - on
            - off
    responses:
      200:
        description: La demande d'envoi de SMS a été comprise par le SMSC
      400:
        description: Les paramètres en entrée sont incorrects
      503:
        description: Le SMSC met trop de temps à répondre
      504:
        description: Erreur de connection au SMSC
    """
    auth = request.authorization
    if auth:
        auth = (auth.username, auth.password)
    params = request.args or {}
    headers = request.headers or {}
    json_doc = request.get_json() or {}
    sender_id = json_doc.get('sender_id', 'bip')
    try:
        queue = CONFIG[sender_id]['account']
    except KeyError:
        queue = CONFIG['bip']['account']
    document = {
        'auth': auth or (),
        'params': params,
        'headers': headers,
        'json': json_doc
        }
    try:
        connection = pika.BlockingConnection(
            pika.ConnectionParameters(
                host='localhost',
                virtual_host='/smsc'
                )
            )
    except Exception:
        # exception de connection lancée par pika
        # il faut que rabbitmq-server soit installé
        # et que le vhost /{service} existe déjà
        return internal_error(
            message="serveur RabbitMQ injoignable",
            code='{}-5000'.format(SERVICE_CODE),
            service_code=SERVICE_CODE
            )
    else:
        channel = connection.channel()
        channel.exchange_declare(
            exchange='sendsms',
            exchange_type='topic'
            )
        # déclaration idempotente: pas grave si l'exchange existe déjà
        channel.basic_publish(
            exchange='sendsms',
            routing_key=queue,
            body=json.dumps(document),
            properties=pika.BasicProperties(delivery_mode=2)
            )
        connection.close()
        return success(service_code=SERVICE_CODE)

# EOF
