#!/usr/bin/env python
# encoding: utf-8

from datetime import datetime
from multiprocessing import Queue
from multiprocessing.queues import Empty

from systemd import journal

from .consts import CONFIG, LOGS

import random

import smpplib.client
import smpplib.command
import smpplib.consts
import smpplib.exceptions
import smpplib.smpp
import smpplib.gsm

import records


QUEUE = Queue()


class Client(object):

    def __init__(self, name='default'):
        self.msisdn = None
        smpp_config = 'SMPP_{}'.format(name)
        self.smpp_client = smpplib.client.Client(
            CONFIG[smpp_config]['host'],
            int(CONFIG[smpp_config]['port'])
            )
        self.smpp_client.set_message_sent_handler(self.callback_request)
        self.smpp_client.set_message_received_handler(self.callback_response)
        self.count = 0
        self.db_engine = CONFIG['db']['engine']
        self.db_user = CONFIG['db']['user']
        self.db_password = CONFIG['db']['password']
        self.db_host = CONFIG['db']['host']
        self.db_port = CONFIG['db']['port']
        self.db_name = CONFIG['db']['database']
        try:
            self.smpp_client.connect()
        except smpplib.exceptions.ConnectionError:
            raise SystemError("Impossible de se connecter au SMSC")
        else:
            try:
                self.smpp_client.bind_transceiver(
                    system_id=CONFIG[smpp_config]['username'],
                    password=CONFIG[smpp_config]['password']
                    )
            except Exception as e:
                LOGS.logger.info('Error:Connect bind_transceiver{}'.format(e))
            else:
                LOGS.logger.info('Connect bind_transceiver')
            from threading import Thread
            t = Thread(target=self.listen)
            t.start()

    def listen(self):
        while True:
            try:
                self.smpp_client.listen()
            except Exception as exc:
                print(self.msisdn, exc)

    def database_url(self, **kwargs):
        url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
            eng=kwargs.get('engine'),
            user=kwargs.get('user'),
            password=kwargs.get('password'),
            host=kwargs.get('host'),
            port=kwargs.get('port'),
            name=kwargs.get('name')
            )
        return url

    def callback_request(self, pdu):
        """Si le SMSC dit qu'il a envoyé le SMS, on compte comme OK tests"""
        if (
            (self.msisdn[-9:]).startswith("34")
            or (self.msisdn[-9:]).startswith("39")
        ):
            pass
        else:
            infos = {
                'date': datetime.now().strftime('%d-%m-%Y %H:%M:%S'),
                'msisdn': self.msisdn,
                'submit': True
                }
            msg_id = pdu.message_id.decode("utf-8")
            message_id = int(str(msg_id), 16)
            kwargs = {
                'engine': self.db_engine,
                'user': self.db_user,
                'password': self.db_password,
                'host': self.db_host,
                'port': self.db_port,
                'name': self.db_name
                }
            url = self.database_url(**kwargs)
            with records.Database(url) as db:
                sql = (
                    "INSERT INTO get_delivery_sms "
                    "(request_id, message_id, status , "
                    "code, msisdn, msisdn_source, submit_date) "
                    "VALUES "
                    "(:request_id, :message_id, :status, "
                    ":code, :msisdn, :msisdn_source, :submit_date)"
                    )
                LOGS.logger.info(sql)
                db.query(
                    sql,
                    request_id=self.request_id,
                    message_id=str(message_id),
                    status='PENDING',
                    code='111',
                    msisdn=self.msisdn[-9:],
                    msisdn_source=self.source[-9:],
                    submit_date=datetime.now().strftime('%y%m%d%H%M')
                    )
            msg = '<<< seq={} msg_id={} msisdn={}'.format(
                pdu.sequence,
                str(message_id),
                self.msisdn
                )
            journal.send(msg, REQUEST_ID=self.request_id)
            QUEUE.put(infos)

    def callback_response(self, pdu):
        """Si le SMSC a envoyé le sms on enregistre """
        """le status du sms dans une tabel"""
        # msisdn_source = pdu.destination_addr.decode("utf-8")
        msisdn_dest = pdu.source_addr.decode("utf-8")
        res = pdu.short_message.decode("utf-8")
        res = res.replace('submit date', 'submit_date')
        res = res.replace('done date', 'done_date')
        res = res.split(' ')
        infos = {
            'date': datetime.now().strftime('%d-%m-%Y %H:%M:%S'),
            'msisdn': msisdn_dest,
            'submit': True
            }
        for x in res:
            val = x.split(':')
            infos.update({val[0]: val[1]})
        cond = (
            infos['msisdn'] is not None
            or infos['msisdn'] != 'None'
            )
        if cond:
            LOGS.logger.info(cond)
            kwargs = {
                'engine': self.db_engine,
                'user': self.db_user,
                'password': self.db_password,
                'host': self.db_host,
                'port': self.db_port,
                'name': self.db_name
                }
            url = self.database_url(**kwargs)
            with records.Database(url) as db:
                sql = (
                    "UPDATE get_delivery_sms SET done_date=:done_date, "
                    "status=:status, code=:code WHERE msisdn=:msisdn "
                    "AND code=:err_code AND message_id=:message_id"
                    )
                LOGS.logger.info(sql)
                if (msisdn_dest[-9:]).startswith("34") or (msisdn_dest[-9:]).startswith("39"):
                    LOGS.logger.info("-----", msisdn_dest, "-----")
                    db.query(
                        sql,
                        message_id=infos['id'][1:],
                        done_date=infos['done_date'].strip(),
                        status="DELIVRD",
                        code=infos['err'],
                        err_code='111',
                        msisdn=msisdn_dest[-9:]
                        )
                else:
                    LOGS.logger.info("-----{}: {}".format(msisdn_dest, infos['stat']))
                    db.query(
                        sql,
                        message_id=infos['id'][1:],
                        done_date=infos['done_date'].strip(),
                        status=infos['stat'],
                        code=infos['err'],
                        err_code='111',
                        msisdn=msisdn_dest[-9:]
                        )
            msg = '<<< msg_id={} status={} msisdn={} code={}'.format(
                infos['id'][1:],
                infos['stat'],
                msisdn_dest,
                infos['err']
                )
            journal.send(msg, REQUEST_ID=self.request_id)
            LOGS.logger.info(infos)

    def send_sms(self, msisdn, message, simulate=False, **kwargs):
        """Envoi du SMS au SMSC"""
        self.msisdn = msisdn
        self.sender_id = kwargs.get('sender_id', 'bip')
        try:
            self.source = CONFIG[self.sender_id]['source']
        except KeyError:
            self.source = CONFIG['bip']['source']
        log_source = '<<< source={}'.format(self.source)
        journal.send(log_source)
        self.request_id = kwargs['request_id']
        self.infos = {
            'msisdn': msisdn,
            'submit': False
            }
        if not simulate:
            if not kwargs.get('intl'):
                destination = '261{}'.format(msisdn[-9:])
            else:
                destination = msisdn
            parts, encoding_flag, msg_type_flag = smpplib.gsm.make_parts(
                message
                )
            # pour l'instant on laisse
            # SMPPLIB tenter du GSM-7, puis du UCS2
            for part in parts:
                pdu = self.smpp_client.send_message(  # NOQA
                    source_addr_ton=0,
                    source_addr_npi=0,
                    source_addr=self.source,
                    dest_addr_ton=0,
                    dest_addr_npi=0,
                    destination_addr=destination,
                    esm_class=msg_type_flag,
                    registered_delivery=1,
                    data_coding=encoding_flag,
                    short_message=part
                    )
                log_msg = '>>> seq={} msisdn={}'.format(
                    pdu.sequence,
                    self.msisdn
                    )
                journal.send(log_msg, REQUEST_ID=self.request_id)
        else:
            choice = random.randint(0, 9)
            if choice % 2:  # nombre impair
                infos = {
                    'date': datetime.now().strftime('%d-%m-%Y %H:%M:%S'),
                    'msisdn': msisdn,
                    'message_id': str(choice),
                    'submit': True
                    }
                QUEUE.put(infos)
        try:
            infos = QUEUE.get(timeout=1)
        except Empty:
            self.infos['date'] = datetime.now().strftime('%d-%m-%Y %H:%M:%S')
            self.infos['message'] = (
                "le SMSC n'a pas donné de réponse à temps"
                )
        else:
            self.infos = infos
            self.count += 1
        return

# EOF
