# -*- coding: utf-8 -*-

from flask import jsonify, request
from datetime import datetime, timedelta

from blueflask.lib.errors import internal_error, bad_request, not_found

from blueflask.lib.infos import success

from ...consts import CONFIG, SERVICE_CODE, LOGS
from . import api
from ... import getsms as lib_customer


import json
import pika


@api.route('/sms', methods=['POST'])
def send_sms():
    """
    Envoi de SMS.
    Cette ressource contacte le SMSC pour envoyer des SMS
    ---
    tags:
      - sms
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: message
        in: body
        required: true
        description: Message court à envoyer
        schema:
          type: string
      - name: msisdn
        in: body
        required: true
        description: Numéro de téléphone destinataire
        schema:
          type: string
      - name: sender_id
        in: body
        required: false
        description: Identifiant SMPP à utiliser
        schema:
          type: string
      - name: simulation
        in: body
        required: false
        description: Interrupteur pour simuler ou non
        schema:
          type: string
          enum:
            - on
            - off
    responses:
      200:
        description: La demande d'envoi de SMS a été comprise par le SMSC
      400:
        description: Les paramètres en entrée sont incorrects
      503:
        description: Le SMSC met trop de temps à répondre
      504:
        description: Erreur de connection au SMSC
    """
    auth = request.authorization
    if auth:
        user = auth.username
        auth = (user, auth.password)
        LOGS.logger.info("{} authenticated".format(user))

    params = request.args or {}
    headers = request.headers or {}
    json_doc = request.get_json() or {}

    LOGS.logger.info(
        "Incoming SMS: Parameters={} headers={} json={}"
        .format(params, headers, json_doc)
    )
    msisdn_sender_id_list = []
    smpp = json_doc.get('sender_id')

    if ',' in json_doc["msisdn"]:
        msisdns = json_doc["msisdn"].replace(' ', '')
        msisdns = msisdns.split(',')

        for msisdn in msisdns:
            sender_id = define_sender_id(msisdn, smpp, user)
            LOGS.logger.info('[{} sent to smpp {}]'.format(msisdn, sender_id))
            msisdn_sender_id_list.append((msisdn, sender_id))
    else:
        sender_id = define_sender_id(json_doc["msisdn"], smpp, user)
        LOGS.logger.info(
            '[{} sent to smpp {}'.format(json_doc["msisdn"], sender_id))
        msisdn_sender_id_list.append((json_doc["msisdn"], sender_id))

    for msisdn, sender_id in msisdn_sender_id_list:
        try:
            json_doc['sender_id'] = sender_id
            json_doc['msisdn'] = msisdn
            queue = CONFIG[sender_id]['account']
        except KeyError:
            return bad_request(
                message="Le sender_id n'exist pas",
                code='{}-{}'.format(
                    SERVICE_CODE,
                    '4001'
                    ),
                service_code=SERVICE_CODE
                )

        document = {
            'auth': auth or (),
            'params': params,
            'headers': headers,
            'json': json_doc
            }

        try:
            connection = pika.BlockingConnection(
                pika.ConnectionParameters(
                    host='localhost',
                    virtual_host='/smsc'
                    )
                )
            LOGS.logger.info('[{}] Connection '
                             'to Pika established'.format(document))
        except Exception:
            # exception de connection lancée par pika
            # il faut que rabbitmq-server soit installé
            # et que le vhost /{service} existe déjà
            return internal_error(
                message="serveur RabbitMQ injoignable",
                code='{}-5000'.format(SERVICE_CODE),
                service_code=SERVICE_CODE
                )
        else:
            channel = connection.channel()
            channel.exchange_declare(
                exchange='sendsms',
                exchange_type='topic'
                )
            # déclaration idempotente: pas grave si l'exchange existe déjà
            channel.basic_publish(
                exchange='sendsms',
                routing_key=queue,
                body=json.dumps(document),
                properties=pika.BasicProperties(delivery_mode=2)
                )
            connection.close()
    return success(service_code=SERVICE_CODE)


@api.route('/sms/<string:uuid>', methods=['GET'])
def get_sms(uuid):
    """
    Lecture des SMS.
    Cette ressource contacte le SMSC pour Lire des SMS
    ---
    tags:
      - sms
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: start
        in: body
        required: false
        description: date de debut de recup des sms
        schema:
          type: date
      - name: stop
        in: body
        required: false
        description: date de fin de recup des sms
        schema:
          type: string
      - name: bool_sms
        in: body
        required: false
        description: check les sms deja lu
        schema:
          type: booleen
    responses:
      200:
        description: La demande de lecture de SMS a été comprise par le SMSC
      400:
        description: Les paramètres en entrée sont incorrects
      503:
        description: Le SMSC met trop de temps à répondre
      504:
        description: Erreur de connection au SMSC
    """
    try:
        customer = lib_customer.Getsms(uuid)
    except ValueError:
        return not_found(
            message="Le compte {} n'existe pas".format(uuid),
            code='{}-{}'.format(
                SERVICE_CODE,
                '4040'
                ),
            service_code=SERVICE_CODE
            )
    else:
        start = request.args.get('start', None)
        stop = request.args.get('stop', None)
        bool_sms = request.args.get('bool_sms', False)
        if not start:
            start = datetime.now() - timedelta(30)
        else:
            try:
                start = datetime.strptime(start, '%d%m%Y %H%M%S')
            except ValueError:
                return bad_request(
                    message='Le format de date '
                    'est ddmmyyyy hhmmss',
                    code='{}-{}'.format(
                        SERVICE_CODE,
                        '4001'
                        ),
                    service_code=SERVICE_CODE
                    )
        if not stop:
            stop = datetime.now()
        else:
            try:
                stop = datetime.strptime(stop, '%d%m%Y %H%M%S')
            except ValueError:
                return bad_request(
                    message='Le format correct de date'
                    ' est ddmmyyyy hhmmss',
                    code='{}-{}'.format(
                        SERVICE_CODE,
                        '4001'
                        ),
                    service_code=SERVICE_CODE
                    )
        infos = customer.check_sms(
            start,
            stop,
            bool_sms
            )
    return jsonify(infos)


@api.route('/sms/delivery', methods=['GET'])
def get_delivery_sms():
    """
    Lecture des SMS.
    Cette ressource contacte le SMSC pour Lire des SMS
    ---
    tags:
      - sms
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: start
        in: body
        required: false
        description: date de debut de recup des sms
        schema:
          type: date
      - name: stop
        in: body
        required: false
        description: date de fin de recup des sms
        schema:
          type: string
      - name: bool_sms
        in: body
        required: false
        description: check les sms deja lu
        schema:
          type: booleen
    responses:
      200:
        description: La demande de lecture de SMS a été comprise par le SMSC
      400:
        description: Les paramètres en entrée sont incorrects
      503:
        description: Le SMSC met trop de temps à répondre
      504:
        description: Erreur de connection au SMSC
    """
    customer = lib_customer.Getsms()
    msisdn = request.args.get('msisdn', None)
    request_id = request.args.get('request_id', None)
    if not request_id:
        return bad_request(
                  message='Le format request_id '
                          'est incorrect ',
                  code='{}-{}'.format(
                      SERVICE_CODE,
                      '4001'
                      ),
                  service_code=SERVICE_CODE
                  )
    else:
        infos = customer.delivery_sms(
            request_id,
            msisdn
            )
    return jsonify(infos)


@api.route('/sms/<string:uuid>', methods=['PUT'])
def update_sms(uuid):
    """
    modifier status de SMS.
    Cette ressource contacte la DB pour modifier le status des SMS
    ---
    tags:
      - sms
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: id
        in: body
        required: false
        description: id du sms
        schema:
          type: int
    responses:
      200:
        description: La demande de lecture de SMS a été comprise par le SMSC
      400:
        description: Les paramètres en entrée sont incorrects
      503:
        description: Le SMSC met trop de temps à répondre
      504:
        description: Erreur de connection au SMSC
    """
    customer = lib_customer.Getsms(uuid)
    ident = request.args.get('ident', None)
    if not ident:
        return bad_request(
            message='Identifiant obligatoire',
            code='{}-{}'.format(
                SERVICE_CODE,
                '4001'
                ),
            service_code=SERVICE_CODE
            )
    else:
        infos = customer.update_sms(ident)
        return jsonify(infos)


@api.route('/sms/connect/<string:uuid>', methods=['POST'])
def connect_sms(uuid):
    """
    Cette ressource ce connecte/deconnecte au SMSC
    ---
    tags:
      - sms
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
    responses:
      200:
        description: La demande de lecture de SMS a été comprise par le SMSC
      400:
        description: Les paramètres en entrée sont incorrects
      503:
        description: Le SMSC met trop de temps à répondre
      504:
        description: Erreur de connection au SMSC
    """
    result = {}
    try:
        customer = lib_customer.Getsms(uuid)
        result = customer.connect_receiver_sms()
    except ValueError:
        return not_found(
            message="Impossible de se reconnecter au SMSC",
            code='{}-{}'.format(
                SERVICE_CODE,
                '4040'
                ),
            service_code=SERVICE_CODE
            )
    return jsonify(result)


def define_sender_id(msisdn, smpp, user):
    telma = (msisdn[-9:]).startswith("34")
    # if smpp == "mfs":
    #     sender_id = "mfs"
    if telma:
        LOGS.logger.info("MSISDN Telma redirected to Orange SMPP.")
        sender_id = "orange"
    elif smpp == "mfs":
        sender_id = "mfs"
    elif not telma and smpp == 'orange':
        LOGS.logger.warning(
            "MSISDN '{}' is not Telma but targeting Orange SMPP. "
            "Redirected to default SMPP.".format(msisdn)
        )
        sender_id = "Blueline"
    else:
        if user.startswith('partenaire'):
            sender_id = user.split('_')[1]
        else:
            sender_id = smpp or 'bip'
    return sender_id
# EOF
