#!/usr/bin/env python
# encoding: utf-8

import json
import threading


import pika
import rest_client
from blueflask.lib.errors import bad_request


from . import LOGS
from . import CONFIG
from .locale import LOCALE_MSG
SERVICE_CODE = '040-05'

# traitement des msg venant de rabbitmq


def process(queue):
    """ Traitement des files d'attentes """
    connection = pika.BlockingConnection(
        pika.ConnectionParameters(
            host='localhost',
            virtual_host='/ussd'
        )
    )

    channel = connection.channel()
    channel.basic_consume(Callback.call, queue=queue)
    channel.start_consuming()


class Worker(threading.Thread):
    def __init__(self, name, arg):
        threading.Thread.__init__(self)
        self.name = name
        self.arg = arg

    def run(self):
        process(self.arg)


class Callback(object):
    def __init__(self, method, **kwargs):
        self.method = method
        self.queue = 'activate.ussd'
        self.channel = kwargs.get('channel')
        self.url_blue = CONFIG['SERVICES']['bluebase']
        self.url_smsc = CONFIG['SERVICES']['smsc']
        self.url_ocs = CONFIG['SERVICES']['ocs']
        self.client_blue = rest_client.RestClient(self.url_blue)
        self.client_smsc = rest_client.RestClient(self.url_smsc)
        self.client_ocs = rest_client.RestClient(self.url_ocs)

    @staticmethod
    def call(ch, method, headers, body):
        LOGS.logger.info(method)
        LOGS.logger.info(headers)
        params = body.decode('utf-8')
        document = json.loads(params)
        # on aura besoin des paramètres suivants
        # pour l'activation
        # et l'envoi d'un sms
        token = document['token']
        params = document['data']
        header_req = document['headers']
        caller_num = params['caller_num']
        operator = params['operator']
        amount = params['amount']
        offer_ref = params['offer_ref']
        # log des content
        # LOGS.logger.debug(document)
        callback = Callback(method, channel=ch)
        # activation de l'offre
        state = callback.activate_offer(token, header_req, **params)
        LOGS.logger.info(state)
        # envoi sms
        # ici on doit faire un choix pour l'envoi du notification
        # selon l'operateur
        if operator == "bip":
            callback.notify_via_sms(
                caller_num,
                amount,
                offer_ref,
                token,
                **state)
        else:
            callback.through_to_redis()
        # fin des tâches
        ch.basic_ack(delivery_tag=method.delivery_tag)

    def activate_offer(self, token, header_req, **params):
        # appel de l'api bluebase
        # pour l'activation
        caller_num = params['caller_num']
        amount = params['amount']
        billing_state = self.billing_ocs(token, header_req, caller_num, amount)
        if billing_state['status'] == 200:
            # faire connaitre à bluebase le solde du client
            params['solde'] = amount
            result = self.client_blue.post(
                '/ussd_buy_recharge',
                auth=(token, ''),
                headers=header_req,
                json=params
                )
            LOGS.logger.info(result)
            assert 'status' in result
            assert 'content' in result
        elif billing_state['status'] == 400:
            result = {
                'status': '400',
                'content': 'la facturation n\'a pas aboutit',
                'headers': header_req
            }
        else:
            result = {
                'status': 401,
                'content': 'solde insuffisant',
                'headers': header_req
                }
        return result

    def through_to_redis(document):
        LOGS.logger.critical(document)

    def notify_via_sms(self, caller_num, amount, offer_ref, token, **result):
        # dépendra de activate_offer
        header_req = result['headers']
        teny = self.locale(caller_num, token, header_req)
        LOGS.logger.info(teny)
        LOGS.logger.info(result)
        if result['status'] == 200 or result['status'] == 204:
            translated_msg = LOCALE_MSG[teny]['200']
            message = "{}: {} {} {}".format(
                translated_msg,
                caller_num,
                offer_ref,
                amount)
            LOGS.logger.info(message)
        elif result['status'] == 401:
            translated_msg = LOCALE_MSG[teny]['401']
            message = "{}: {} {} {}".format(
                translated_msg,
                caller_num,
                offer_ref,
                amount)
            LOGS.logger.info(message)
        else:
            translated_msg = LOCALE_MSG[teny]['200']
            message = "{}: {} {} {}".format(
                translated_msg,
                caller_num,
                offer_ref,
                amount)
            # ici je fait un rollback pour ocs
            self.rollback_ocs(token, caller_num, header_req, amount)
            LOGS.logger.info(message)
        # une fois le message obtenu
        # on relai le message à l'api de smsc
        # en mode asynchrone
        data_sms = {'msisdn': caller_num, 'message': message}
        result = self.client_smsc.post(
            '/async/sms',
            auth=(token, ''),
            headers=header_req,
            json=data_sms
             )
        if result['status'] != 200:
            LOGS.logger.critical("echec envoi sms")

    def billing_ocs(self, token, header_req, caller_num, amount):
        phonenumber = caller_num
        # récupération solde actuel
        try:
            account = self.client_ocs.get(
                '/subscribers/{phonenumber}/balances'.format(
                    phonenumber=phonenumber
                    ),
                auth=(token, ''),
                headers=header_req,
                json={'account_type': '2000'}
                )
            LOGS.logger.info(account['content']['message'])
            ocs_data = account['content']['message']
            balance = ocs_data[0]
        except SyntaxError as exc:
            return bad_request(
                message="paramètres incomplets",
                code='040-05-400'
            )
        # debiter ocs
        if int(balance) >= int(amount):
            # activate_offer
            # maj account ocs
            due = -1 * int(amount)
            new_balance = int(balance) + due
            account_new = self.client_ocs.put(
                '/subscribers/{phonenumber}/balances'.format(
                    phonenumber=phonenumber
                ),
                auth=(token, ''),
                headers=header_req,
                json={
                    'amount': due,
                    'operation_type': '2',
                    'account_type': 2000
                    }
            )
            # verifier si la requête s'est bien passée
            if account_new['status'] == 200 or account_new['status'] == 204:
                billing_state = {
                    'status': 200,
                    'balance': new_balance
                    }
            else:
                billing_state = {
                    'status': 400,
                    'balance': balance
                }
            return billing_state
        else:
            billing_state = {
                'status': 401,
                'balance': balance
            }
            return billing_state

    def rollback_ocs(self, token, caller_num, header_req, amount):
        phonenumber = caller_num
        account = self.client_ocs.put(
                '/subscribers/{phonenumber}/balances'.format(
                    phonenumber=phonenumber
                ),
                auth=(token, ''),
                headers=header_req,
                json={
                    'amount': int(amount),
                    'operation_type': '2',
                    'account_type': 2000
                    }
            )
        if account['status'] == 200 or account['status'] == 204:
            LOGS.logger.info("rollback effectué: {}".format(account))
        else:
            LOGS.logger.critical("echec rollback: {}"
                                 .format(account['content']))

    # Récuperation langue
    def locale(self, caller_num, token, header_req):
        lang = self.client_ocs.get(
            '/subscribers/{caller_num}'.format(caller_num=caller_num),
            auth=(token, ''),
            headers=header_req,
            params={
                'key': 'language_code'
                }
            )
        lang_code = lang['content']['message']
        if lang_code['language_code'] == 3:
            locale = 'mg'
        elif lang_code['language_code'] == 2:
            locale = 'fr'
        elif lang_code['language_code'] == 1:
            locale = 'en'
        return locale
