#!/usr/bin/env python
# encoding: utf-8
import json

import rest_client
import pika

from . import CONFIG
from . import LOGS
from .core import check_parameters
SERVICE_CODE = '040-05'


class Bluebase():
    """
    api pour 4D
    """

    def __init__(self):
        self.url = CONFIG['SERVICES']['bluebase']
        self.client = rest_client.RestClient(self.url)

    @check_parameters('caller_num', 'login_type', 'login')
    def authentication(self, token, **data):
        """
        Cette méthode permet de vérifier l'existence d'un client
        et de récupérer ses abonnements suivant le type de service
        passé en paramètre.
        """
        request = data['request']
        data['type_auth'] = data['login_type']
        data['login_auth'] = data['login']
        del data['login']
        del data['login_type']
        del data['request']
        numero = data['caller_num']
        if data['type_auth'] in ['internet_num', 'bip_num']:
            if len(data['login_auth']) != 12:
                data['login_auth'] = data['login_auth'][-9:]
            elif data['login_auth'].startswith('261'):
                data['login_auth'] = data['login_auth'][-9:]
        response = self.client.post(
            '/auth_ussd',
            auth=(token, ''),
            headers=request.headers,
            json=data
            )
        if response['status'] == 200:
            blue_data = response['content']['root']['data']
            # pour le cas d'un revendeur
            if numero.startswith('03900') or numero.startswith('3900'):
                blue_data['num_is_mine'] = 'Vrai'
            result = {
                'status': response['status'],
                'data': {
                    'customer_id': blue_data['client_refnum'],
                    'name': blue_data['client_nom'],
                    'last_name': blue_data['client_prenom'],
                    'device_name': [],
                    'num_is_mine': blue_data['num_is_mine']
                    }
                }
            if 'device_id' in blue_data:
                if isinstance(blue_data['device_id'], str):
                    # pour le cas où on utilise le device id
                    # pour l'authentification
                    # on ne renvoi pas tous les device
                    LOGS.logger.info(data['type_auth'])
                    if data['type_auth'] == "internet_num" or\
                       data['type_auth'] == "tv_card_num":
                        result['data']['device_name']\
                                .append(data['login_auth'])
                    else:
                        result['data']['device_name'] = [
                            blue_data['device_id']
                            ]
                else:
                    result['data']['device_name'] = blue_data['device_id']
            return result
        else:
            return {'status': response['status'], 'data': response['content']}

    @check_parameters('caller_num', 'customer_id')
    def subscriptions(self, token, **data):
        """
        Cette méthode permet de récupérer les informations relatives à
        un abonnement donné.
        """
        request = data['request']
        data['client_refnum'] = data['customer_id']
        data['device_id'] = data['device_name']
        del data['customer_id']
        del data['request']
        del data['device_name']
        if data['service_type'] == 'internet':
            if len(data['device_id']) != 12:
                data['device_id'] = data['device_id'][-9:]
            elif data['device_id'].startswith('261'):
                data['device_id'] = data['device_id'][-9:]
        response = self.client.post(
            '/ussd_get_infos',
            auth=(token, ''),
            headers=request.headers,
            json=data
            )
        if response['status'] == 200:
            response_data = response['content']['root']['data']
            subscriptions = response_data['bouquet']
            if not isinstance(subscriptions, list):
                subscriptions = [subscriptions]
            subscriptions = [
                {
                    'name': subscription['intitule'],
                    'value': subscription['date_fin']
                    }
                for subscription in subscriptions
                ]
            result = {
                'status': response['status'],
                'data': subscriptions,
                'num_is_mine': response_data['num_is_mine'],
                'caller_num': data['caller_num']
                }
            return result
        else:
            return {'status': response['status'], 'data': response['content']}

    @check_parameters('caller_num', 'customer_id', 'balance')
    def offers(self, token, **data):
        """
        Cette méthode permet de récupérer les recharges qu'un client
        peut s'acheter en fonction du solde passé en paramètre.
        """
        request = data['request']
        data['client_refnum'] = data['customer_id']
        data['solde'] = data['balance']
        data['device_id'] = data['device_name']
        del data['request']
        del data['customer_id']
        del data['balance']
        del data['device_name']
        if data['service_type'] == 'internet':
            if len(data['device_id']) != 12:
                data['device_id'] = data['device_id'][-9:]
            elif data['device_id'].startswith('261'):
                data['device_id'] = data['device_id'][-9:]
        response = self.client.post(
            '/ussd_get_recharge',
            auth=(token, ''),
            headers=request.headers,
            json=data
            )
        if response['status'] == 200:
            data = response['content']['root']['data']
            offers = data['offre']
            if not isinstance(offers, list):
                offers = [offers]
            offers = [
                {
                    'name': offer['intitule'],
                    'price': offer['price'],
                    'offer_id': offer['refnum']
                    }
                for offer in offers
                ]
            result = {
                'status': response['status'],
                'data': offers
                }
            return result
        else:
            return {'status': response['status'], 'data': response['content']}

    # traitement asynchrone des activations de l'offre
    # producer definition
    # à implementer plus proprement aprés
    def process(self, message):
        connection = pika.BlockingConnection(
            pika.ConnectionParameters(
                host='localhost',
                virtual_host='/ussd'
                )
            )
        channel = connection.channel()
        channel.exchange_declare(
            exchange='ussd_offer',
            exchange_type='topic'
            )
        channel.basic_publish(
            routing_key='ussd_msg',
            exchange='ussd_offer',
            properties=pika.BasicProperties(
                app_id='ussd-services',
                delivery_mode=2,  # rendre le message persistant
                ),
            body=message
            )
        connection.close()

    @check_parameters(
        'caller_num',
        'customer_id',
        'offer_id',
        'device_name',
        'offer_ref'
    )
    def buy_recharge(self, token, **data):
        """
        Cette méthode permet de placer un ordre d'achat de recharge.
        l'activation sera effectuée en mode asyncrhone
        """
        request = data['request']
        data['device_id'] = data['device_name']
        del data['request']
        del data['device_name']
        if data['service_type'] == 'internet':
            if len(data['device_id']) != 12:
                data['device_id'] = data['device_id'][-9:]
            elif data['device_id'].startswith('261'):
                data['device_id'] = data['device_id'][-9:]
        result = {
            'data': {
                'client_refnum': data['customer_id'],
                'offre_refnum': data['offer_id'],
                'amount': data['amount'],
                'device_id': data['device_id'],
                'caller_num': data['caller_num'],
                'operator': data['operator'],
                'offer_ref': data['offer_id'],
                'service_type': data['service_type']
                },
            'status': '200',
            'headers': request.headers,
            'token': token
            }
        LOGS.logger.info(result)
        message = json.dumps(result)
        self.process(message)
        return {'status': result['status']}
